from itertools import chain
from typing import ClassVar, Iterable, Optional, Set, Tuple, Union

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.enums import Role
from mail.payments.payments.core.entities.manager import Manager
from mail.payments.payments.core.exceptions import ManagerActionNotAllowed
from mail.payments.payments.storage.exceptions import ManagerNotFound

OptRoleTuple = Union[Tuple[Role, ...], Tuple[()]]


class BaseManagerAction(BaseDBAction):
    """Base action to be run by manager.
    Requires one of `require_roles` (or higher) privileges.
    Does not check for roles if `require_roles` is empty.

    Loads manager into `self.manager`
    """

    require_roles: ClassVar[OptRoleTuple] = ()

    manager: Optional[Manager]

    def __init__(self, manager_uid: int):
        super().__init__()
        self.manager_uid: int = manager_uid

    @staticmethod
    def _with_subroles(roles: Iterable[Role]) -> Set[Role]:
        """Returns a set of subroles given `roles` resolve into"""
        return set((
            subrole
            for role in roles
            for subrole in chain((role,), role.subroles)
        ))

    def action_allowed(self, roles: Iterable[Role]) -> bool:
        if not self.require_roles:  # roles not required
            return True
        roles = self._with_subroles(roles)
        return bool(set(self.require_roles).intersection(roles))

    async def _check_manager_roles(self, manager_uid: int) -> None:
        """Check if manager has enough privileges to run action"""
        try:
            self.manager = await self.storage.manager.get(uid=manager_uid)
        except ManagerNotFound:
            raise ManagerActionNotAllowed(f'User {self.manager_uid} is not authorized')
        roles = [r.role async for r in self.storage.manager_role.find(manager_uid=self.manager.uid)]
        if not self.action_allowed(roles):
            raise ManagerActionNotAllowed(
                f'Manager {self.manager.uid} is not allowed to run {self.__class__.__name__} action'
            )

    async def pre_handle(self) -> None:
        await self._check_manager_roles(self.manager_uid)
        assert self.manager is not None
        self.logger.context_push(manager_uid=self.manager.uid)
        await super().pre_handle()
