from datetime import datetime
from typing import Any, ClassVar, Dict, Iterable, List, Optional, Tuple

from mail.payments.payments.core.actions.manager.base import BaseManagerAction
from mail.payments.payments.core.actions.merchant.block import BlockMerchantAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.mixins.moderation import MerchantModerationMixin
from mail.payments.payments.core.actions.moderation import ScheduleMerchantModerationAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.common import SearchStats
from mail.payments.payments.core.entities.enums import (
    AcquirerType, FunctionalityType, MerchantStatus, ModerationStatus, OperationKind, Role
)
from mail.payments.payments.core.entities.keyset import Keyset, KeysetEntry, ManagerMerchantListKeysetEntity
from mail.payments.payments.core.entities.merchant import Merchant, MerchantsAdminData
from mail.payments.payments.core.entities.moderation import Moderation
from mail.payments.payments.core.entities.userinfo import UserInfo
from mail.payments.payments.core.exceptions import KeysetInvalidError, ModerationAlreadyExistsError, SortByInvalidError
from mail.payments.payments.interactions.exceptions import BlackBoxUserNotFoundError


class GetMerchantListManagerAction(BaseManagerAction):
    require_roles = (Role.ASSESSOR,)
    SORT_BY_ACCEPTABLE: ClassVar[Tuple[str, ...]] = ('uid', 'created', 'updated')
    SORT_BY_DEFAULT = 'uid'

    def __init__(self,
                 manager_uid: int,
                 merchant_uid: Optional[int] = None,
                 name: Optional[str] = None,
                 username: Optional[str] = None,
                 client_id: Optional[str] = None,
                 submerchant_id: Optional[str] = None,
                 limit: Optional[int] = None,
                 offset: Optional[int] = None,
                 sort_by: str = SORT_BY_DEFAULT,
                 desc: Optional[bool] = None,
                 acquirers: Optional[Iterable[AcquirerType]] = None,
                 created_from: Optional[datetime] = None,
                 created_to: Optional[datetime] = None,
                 updated_from: Optional[datetime] = None,
                 updated_to: Optional[datetime] = None,
                 moderation_status: Optional[ModerationStatus] = None,
                 keyset: Optional[ManagerMerchantListKeysetEntity] = None,
                 ):
        super().__init__(manager_uid=manager_uid)
        self.merchant_uid: Optional[int] = merchant_uid
        self.name: Optional[str] = name
        self.username: Optional[str] = username
        self.client_id: Optional[str] = client_id
        self.submerchant_id: Optional[str] = submerchant_id
        self.limit: Optional[int] = limit
        self.offset: Optional[int] = offset
        self.sort_by: str = sort_by
        self.desc: Optional[bool] = desc
        self.statuses: Optional[Iterable[MerchantStatus]] = [_ for _ in MerchantStatus if _ != MerchantStatus.DRAFT]
        self.acquirers: Optional[Iterable[AcquirerType]] = acquirers
        self.created_from: Optional[datetime] = created_from
        self.created_to: Optional[datetime] = created_to
        self.updated_from: Optional[datetime] = updated_from
        self.updated_to: Optional[datetime] = updated_to
        self.moderation_status: Optional[ModerationStatus] = moderation_status
        self.keyset: Optional[ManagerMerchantListKeysetEntity] = keyset

    def _make_next_page_keyset(self, merchants: List[Merchant]) -> Optional[ManagerMerchantListKeysetEntity]:
        if not merchants:
            return None
        sort_by = []
        if self.keyset is not None:
            for name in self.keyset.sort_order:
                entry = getattr(self.keyset, name)
                order = entry.order
                sort_by.append((name, order))
        else:
            order = 'desc' if self.desc else 'asc'
            sort_by.append((self.sort_by, order))
            if self.sort_by != 'uid':
                sort_by.append(('uid', order))

        keyset_entries = {}
        for column, order in sort_by:
            func = max if order == 'asc' else min
            barrier = func(getattr(item, column) for item in merchants)
            keyset_entry = KeysetEntry(order=order, barrier=barrier)

            keyset_entries[column] = keyset_entry

        return ManagerMerchantListKeysetEntity(sort_order=[item[0] for item in sort_by], **keyset_entries)

    def _get_keyset_filter(self) -> Optional[Keyset]:
        if self.keyset is None:
            return None
        keyset: Keyset = []
        for name in self.keyset.sort_order:
            entry: KeysetEntry = getattr(self.keyset, name)
            if name != self.sort_by and name != 'uid':
                raise KeysetInvalidError

            keyset.append((name, entry.order, entry.barrier))

        return keyset

    async def _get_merchants(self) -> Tuple[List[Merchant], SearchStats]:
        if self.username is not None:
            # Резолвим username в blackbox, полученный uid затем используем для фильтрации.
            # Это надёжнее, чем брать username из json поля в базе - ведь blackbox за нас учтёт всю их
            # бизнес-логику с отождествлением логинов (напр. иногда дефис и точка в логине взаимозаменяемы)
            user_info: Optional[UserInfo] = None
            try:
                user_info = await self.clients.blackbox.userinfo(login=self.username)
            except BlackBoxUserNotFoundError:
                pass

            if user_info is None or self.merchant_uid is not None and user_info.uid != self.merchant_uid:
                return [], SearchStats(total=0, found=0)
            self.merchant_uid = user_info.uid

        filter_params: Dict[str, Any] = {
            'uid': self.merchant_uid,
            'name': self.name,
            'client_id': self.client_id,
            'submerchant_id': self.submerchant_id,
            'statuses': self.statuses,
            'acquirers': self.acquirers,
            'created_from': self.created_from,
            'created_to': self.created_to,
            'updated_from': self.updated_from,
            'updated_to': self.updated_to,
            'moderation_status': self.moderation_status,
        }

        sort_by: Optional[str] = self.sort_by
        keyset = self._get_keyset_filter()
        if keyset is not None:
            sort_by = None
        # Getting from db
        merchants = [
            m
            async for m in self.storage.merchant.find(
                limit=self.limit,
                offset=self.offset,
                sort_by=sort_by,
                descending=self.desc,
                keyset=keyset,
                **filter_params,
            )
        ]
        # Loading
        merchants = [
            await GetMerchantAction(merchant=m).run()
            for m in merchants
        ]
        found = await self.storage.merchant.get_found_count(**filter_params)
        total = await self.storage.merchant.get_found_count(statuses=self.statuses)
        return merchants, SearchStats(total=total, found=found)

    async def handle(self) -> MerchantsAdminData:
        if self.sort_by not in self.SORT_BY_ACCEPTABLE:
            raise SortByInvalidError
        merchants, search_stats = await self._get_merchants()

        keyset = self._make_next_page_keyset(merchants)

        return MerchantsAdminData(
            merchants=merchants,
            stats=search_stats,
            keyset=keyset,
        )


class BlockMerchantManagerAction(BaseManagerAction):
    transact = True
    require_roles = (Role.ADMIN,)

    def __init__(self,
                 manager_uid: int,
                 uid: int,
                 block: bool = True,
                 terminate_contract: bool = True,
                 ):
        super().__init__(manager_uid=manager_uid)
        self.uid: int = uid
        self.block: bool = block
        self.terminate_contract: bool = terminate_contract

    async def handle(self):
        return await BlockMerchantAction(uid=self.uid,
                                         block=self.block,
                                         terminate_contract=self.terminate_contract).run()


class RecreateMerchantModerationManagerAction(BaseManagerAction, MerchantModerationMixin):
    require_roles = (Role.ASSESSOR,)

    def __init__(self,
                 manager_uid: int,
                 uid: int
                 ):
        super().__init__(manager_uid=manager_uid)
        self.uid = uid

    async def handle(self) -> Moderation:
        merchant = await GetMerchantAction(uid=self.uid).run()
        if await self.approved_effective_moderation(merchant, functionality_type=FunctionalityType.PAYMENTS):
            raise ModerationAlreadyExistsError
        # save merchant in order to update revision
        merchant = await self.storage.merchant.save(merchant)
        self.logger.context_push(revision=merchant.revision)
        moderation = await ScheduleMerchantModerationAction(
            merchant=merchant, functionality_type=FunctionalityType.PAYMENTS,
        ).run()
        await self.storage.change_log.create(ChangeLog(
            uid=merchant.uid,
            revision=merchant.revision,
            operation=OperationKind.EDIT_MERCHANT,
            arguments={
                'manager_uid': self.manager_uid,
                'uid': self.uid,
                'moderation_id': moderation.moderation_id
            }
        ))
        return moderation


class UpdateSupportCommentManagerAction(BaseManagerAction):
    transact = True
    require_roles = (Role.ASSESSOR,)

    def __init__(self,
                 manager_uid: int,
                 uid: int,
                 support_comment: str
                 ):
        super().__init__(manager_uid=manager_uid)
        self.uid: int = uid
        self.support_comment: str = support_comment

    async def handle(self):
        merchant = await GetMerchantAction(uid=self.uid, skip_data=True, for_update=True).run()
        merchant.support_comment = self.support_comment
        merchant = await self.storage.merchant.save(merchant)
        self.logger.context_push(revision=merchant.revision)
        await self.storage.change_log.create(ChangeLog(
            uid=merchant.uid,
            revision=merchant.revision,
            operation=OperationKind.EDIT_MERCHANT,
            arguments={
                'support_comment': self.support_comment
            }
        ))
        return merchant
