from typing import Any, Optional

from mail.payments.payments.core.actions.manager.base import BaseManagerAction
from mail.payments.payments.core.actions.order.get_list import CoreGetOrderListAction, OrderListResult
from mail.payments.payments.core.entities.common import SearchStats
from mail.payments.payments.core.entities.enums import Role
from mail.payments.payments.core.entities.order import OrdersAdminData
from mail.payments.payments.http_helpers.crypto import DecryptionError, InvalidToken


class GetOrderListManagerAction(BaseManagerAction):
    require_roles = (Role.ASSESSOR,)

    def __init__(self, manager_uid: int, order_hash: Optional[str] = None, **kwargs: Any):
        super().__init__(manager_uid=manager_uid)
        self.order_hash: Optional[str] = order_hash
        self.kwargs: dict = kwargs

    async def handle(self) -> OrdersAdminData:
        if self.order_hash is not None:
            try:
                with self.crypto.decrypt(self.order_hash) as data:
                    self.kwargs['uid'] = data['uid']
                    self.kwargs['order_id'] = data['order_id']
            except (DecryptionError, InvalidToken) as err:
                self.logger.exception(err)
                return OrdersAdminData(orders=[], stats=SearchStats(total=0, found=0))

        self.kwargs['count_found'] = True
        orders: OrderListResult = await CoreGetOrderListAction(**self.kwargs).run()

        # total = await self.storage.order.get_found_count()
        total = 0  # OPLATASUPPORT-71, PAYBACK-917

        found = orders.total_found_count
        assert found is not None, "Core Order List action returned list without total found counter"
        return OrdersAdminData(
            orders=orders,
            stats=SearchStats(total=total, found=found),
        )
