from collections import defaultdict
from typing import Any, Dict, List

from sendr_interactions import exceptions as interaction_errors
from sendr_utils import alist

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.manager.base import BaseManagerAction
from mail.payments.payments.core.entities.enums import Role
from mail.payments.payments.core.entities.manager import Manager, ManagerRole
from mail.payments.payments.core.exceptions import UIDNotFound
from mail.payments.payments.storage.exceptions import ManagerNotFound, ManagerRoleNotFound


class BaseRoleManagerAction(BaseDBAction):
    uid: int

    def __init__(self, domain_login: str):
        super().__init__()
        self.domain_login: str = domain_login

    async def pre_handle(self) -> None:
        await super().pre_handle()
        try:
            userinfo = await self.clients.blackbox_corp.userinfo(login=self.domain_login)
            self.uid = userinfo.uid
        except interaction_errors.InteractionResponseError:
            raise UIDNotFound(message='Unable to get UID for given domain login to assign role')


class AddRoleManagerAction(BaseRoleManagerAction):
    """
    Idempotent creation of Manager and corresponding ManagerRole - if one of them does not exist
    It should be OK, if role already exists - just do nothing
    """
    transact = True

    def __init__(self, domain_login: str, role: Role):
        super().__init__(domain_login=domain_login)
        self.role: Role = role

    async def handle(self) -> ManagerRole:
        try:
            await self.storage.manager.get(uid=self.uid)
        except ManagerNotFound:
            await self.storage.manager.create(Manager(uid=self.uid, domain_login=self.domain_login))
        try:
            manager_role = await self.storage.manager_role.get(manager_uid=self.uid, role=self.role)
        except ManagerRoleNotFound:
            manager_role = await self.storage.manager_role.create(ManagerRole(manager_uid=self.uid, role=self.role))
        return manager_role


class RemoveRoleManagerAction(BaseRoleManagerAction):
    """
    Remove ManagerRole if exists, otherwise do nothing
    """
    def __init__(self, domain_login: str, role: Role):
        super().__init__(domain_login=domain_login)
        self.role: Role = role

    async def handle(self):
        await self.storage.manager_role.delete_for_manager(manager_uid=self.uid, role=self.role)


class ListAllManagerRolesAction(BaseDBAction):
    """List all manager roles"""
    async def handle(self) -> List[Dict[str, Any]]:
        # we do not expect to have a lot of managers, so we get all at once and then fetch their roles
        managers = [m async for m in self.storage.manager.find()]
        roles = [role async for role in self.storage.manager_role.find()]

        uid_to_roles: Dict[int, List[ManagerRole]] = defaultdict(list)
        for role in roles:
            uid_to_roles[role.manager_uid].append(role)

        manager_roles_description = []
        for manager in managers:
            role_description = {
                'login': manager.domain_login,
                'roles': [
                    {'role': role.role.value}
                    for role in uid_to_roles[manager.uid]
                ]
            }
            manager_roles_description.append(role_description)
        return manager_roles_description


class GetManagerRoleAction(BaseManagerAction):
    def __init__(self,
                 manager_uid: int,
                 uid: int,
                 ):
        super().__init__(manager_uid=manager_uid)
        self.uid = uid

    async def handle(self) -> List[Role]:
        db_roles = await alist(self.storage.manager_role.find(manager_uid=self.uid))
        roles = [role.role for role in db_roles]
        for role in roles:
            roles.extend(role.subroles)
        return list(set(roles))
