from datetime import datetime
from typing import Any, Dict, Iterable, Optional

from mail.payments.payments.core.actions.manager.base import BaseManagerAction
from mail.payments.payments.core.entities.common import SearchStats
from mail.payments.payments.core.entities.enums import PayStatus, Role, TransactionStatus
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.transaction import TransactionsAdminData


class GetTransactionListManagerAction(BaseManagerAction):
    require_roles = (Role.ASSESSOR,)

    def __init__(
        self,
        manager_uid: int,
        tx_id: Optional[int] = None,
        statuses: Optional[Iterable[TransactionStatus]] = None,
        uid: Optional[int] = None,
        order_id: Optional[int] = None,
        email: Optional[str] = None,
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        sort_by: Optional[str] = None,
        desc: Optional[bool] = None,
        order_pay_statuses: Optional[Iterable[PayStatus]] = None,
        lower_created_dt: Optional[datetime] = None,
        upper_created_dt: Optional[datetime] = None,
        lower_updated_dt: Optional[datetime] = None,
        upper_updated_dt: Optional[datetime] = None,
        customer_uid: Optional[int] = None,
        services: Optional[Iterable[int]] = None,
    ):
        super().__init__(manager_uid=manager_uid)
        self.tx_id: Optional[int] = tx_id
        self.statuses: Optional[Iterable[TransactionStatus]] = statuses
        self.uid: Optional[int] = uid
        self.order_id: Optional[int] = order_id
        self.email: Optional[str] = email
        self.limit: Optional[int] = limit
        self.offset: Optional[int] = offset
        self.sort_by: Optional[str] = sort_by
        self.desc: Optional[bool] = desc
        self.order_pay_statuses: Optional[Iterable[PayStatus]] = order_pay_statuses
        self.lower_created_dt: Optional[datetime] = lower_created_dt
        self.upper_created_dt: Optional[datetime] = upper_created_dt
        self.lower_updated_dt: Optional[datetime] = lower_updated_dt
        self.upper_updated_dt: Optional[datetime] = upper_updated_dt
        self.customer_uid: Optional[int] = customer_uid
        self.services: Optional[Iterable[int]] = services

    async def handle(self) -> TransactionsAdminData:
        filter_params: Dict[str, Any] = {
            'uid': self.uid,
            'tx_id': self.tx_id,
            'order_id': self.order_id,
            'statuses': self.statuses,
            'order_pay_statuses': self.order_pay_statuses,
            'email_query': self.email,
            'created_from': self.lower_created_dt,
            'created_to': self.upper_created_dt,
            'updated_from': self.lower_updated_dt,
            'updated_to': self.upper_updated_dt,
            'customer_uid': self.customer_uid,
            'services': self.services,
        }

        transactions = []
        async for tx in self.storage.transaction.find(
            sort_by=self.sort_by,
            descending=self.desc,
            limit=self.limit,
            offset=self.offset,
            **filter_params,
        ):
            assert isinstance(tx.order, Order)
            tx.set_trust_receipt_urls(tx.order.acquirer)
            transactions.append(tx)

        total = await self.storage.transaction.get_transactions_count()
        found = await self.storage.transaction.get_found_count(**filter_params)
        return TransactionsAdminData(
            transactions=transactions,
            stats=SearchStats(total=total, found=found)
        )
