from typing import Optional

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.contract import TerminateContractAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.merchant import Merchant


class BlockMerchantAction(BaseMerchantAction):
    """Block merchant, optionally, terminate contract"""

    for_update = True
    transact = True
    skip_data = True
    skip_parent = True

    def __init__(self,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 terminate_contract: bool = True,
                 block: bool = True,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.terminate_contract = terminate_contract
        self.block = block

    async def _block(self):
        self.merchant.blocked = self.block
        await self.storage.merchant.save(self.merchant)
        change_log = ChangeLog(uid=self.merchant.uid,
                               revision=self.merchant.revision,
                               operation=OperationKind.EDIT_MERCHANT,
                               arguments={
                                   'blocked': self.merchant.blocked,
                               })
        await self.storage.change_log.create(change_log)
        self.logger.context_push(merchant_uid=self.merchant.uid)
        message = 'Merchant blocked' if self.block else 'Merchant unblocked'
        self.logger.info(message)

    async def handle(self) -> Merchant:
        assert self.merchant
        await self._block()
        if self.block and self.terminate_contract:
            await TerminateContractAction(merchant=self.merchant).run()
            self.logger.info('Contract terminated')
        return self.merchant
