from typing import Iterable, Optional

from sendr_utils import json_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.init_client import InitClientAction
from mail.payments.payments.core.actions.init_submerchant import InitSubmerchantAction
from mail.payments.payments.core.actions.merchant.check_roles import CheckUpdateMerchantRolesAction
from mail.payments.payments.core.actions.merchant.check_uid_match import CheckUIDMatchAction
from mail.payments.payments.core.actions.merchant.create_entity import CreateMerchantEntityAction
from mail.payments.payments.core.actions.merchant.functionality import put_merchant_functionality
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.enums import (
    AcquirerType, FunctionalityType, MerchantDraftPolicy, MerchantStatus, OrderSource, PersonType
)
from mail.payments.payments.core.entities.functionality import FunctionalityData
from mail.payments.payments.core.entities.merchant import (
    AddressData, BankData, Merchant, MerchantData, OrganizationData, PersonData
)
from mail.payments.payments.core.exceptions import InnModificationError, TinkoffMerchantInvalidDataError
from mail.payments.payments.utils.const import (
    ENTREPRENEUR_PREFIX_FULL, ENTREPRENEUR_PREFIX_SHORT, MAX_LENGTH_ENGLISH_NAME
)
from mail.payments.payments.utils.helpers import (
    fio_str_full, fio_str_short, is_entrepreneur_by_inn, transliterate_to_eng
)


class CreateMerchantAction(BaseMerchantAction):
    transact = True
    allow_none = True
    skip_data = True
    for_update = True
    draft_policy = MerchantDraftPolicy.MERCHANT_DRAFT_ALLOWED

    def __init__(self,
                 name: str,
                 addresses: Iterable[dict],
                 bank: dict,
                 organization: dict,
                 functionality: FunctionalityData,
                 persons: Iterable[dict],
                 username: Optional[str] = None,
                 fast_moderation: bool = False,
                 skip_init_submerchant: bool = False,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.name: str = name
        self.addresses: Iterable[dict] = addresses
        self.bank: dict = bank
        self.organization: dict = organization
        self.persons: Iterable[dict] = persons
        self.username: Optional[str] = username
        self.fast_moderation: bool = fast_moderation
        self.skip_init_submerchant: bool = skip_init_submerchant
        self.functionality = functionality

    @staticmethod
    def _map_merchant_data(addresses: Iterable[dict],
                           bank: dict,
                           organization: dict,
                           persons: Iterable[dict],
                           username: Optional[str] = None,
                           fast_moderation: bool = False,
                           ) -> MerchantData:
        return MerchantData(
            addresses=[
                AddressData(
                    type=address['type'],
                    city=address['city'],
                    country=address['country'],
                    home=address.get('home'),
                    street=address['street'],
                    zip=address['zip'],
                )
                for address in addresses
            ],
            bank=BankData(
                account=bank['account'],
                bik=bank['bik'],
                correspondent_account=bank['correspondent_account'],
                name=bank['name'],
            ),
            organization=OrganizationData(
                type=organization['type'],
                name=organization.get('name'),
                english_name=organization.get('english_name'),
                full_name=organization.get('full_name'),
                inn=organization['inn'],
                kpp=organization['kpp'],
                ogrn=organization['ogrn'],
                schedule_text=organization['schedule_text'],
                site_url=organization.get('site_url'),
                description=organization.get('description'),
            ),
            persons=[
                PersonData(
                    type=person['type'],
                    name=person['name'],
                    email=person['email'],
                    phone=person['phone'],
                    surname=person['surname'],
                    patronymic=person.get('patronymic'),
                    birth_date=person['birth_date'],
                )
                for person in persons
            ],
            username=username,
            fast_moderation=fast_moderation,
        )

    @staticmethod
    def _fill_missing_organization_names(data: MerchantData) -> None:
        assert data.organization and data.organization.inn
        assert data.persons

        if not is_entrepreneur_by_inn(data.organization.inn):
            return

        ceo = next(filter(lambda person: person.type == PersonType.CEO, data.persons), None)
        assert ceo

        organization = data.organization

        if not organization.full_name:
            full_name = fio_str_full(ceo.surname, ceo.name, ceo.patronymic)
            if full_name:
                organization.full_name = ENTREPRENEUR_PREFIX_FULL + full_name

        if not organization.name or not organization.english_name:
            short_name = fio_str_short(ceo.surname, ceo.name, ceo.patronymic)
            if short_name:
                short_name = ENTREPRENEUR_PREFIX_SHORT + short_name
                if not organization.name:
                    organization.name = short_name
                if not organization.english_name:
                    organization.english_name = transliterate_to_eng(short_name)[0:MAX_LENGTH_ENGLISH_NAME]

    async def _init_payments_functionality(self) -> None:
        assert self.uid is not None
        await put_merchant_functionality[FunctionalityType.PAYMENTS](
            merchant=self.merchant,
            data=self.functionality,
        ).run()
        # Initializing tinkoff entities
        try:
            if not self.skip_init_submerchant:
                await InitSubmerchantAction(merchant=self.merchant, save=False).run()
        except TinkoffMerchantInvalidDataError as e:
            if not settings.TINKOFF_IGNORE_MERCHANT_CREATE_FAIL:
                raise e

        # Initializing balance entites
        await InitClientAction(merchant=self.merchant, save=False).run()

    async def _init_yandex_pay_functionality(self) -> None:
        assert self.uid is not None
        await put_merchant_functionality[FunctionalityType.YANDEX_PAY](
            merchant=self.merchant,
            data=self.functionality,
        ).run()

    async def handle(self) -> Merchant:
        assert self.uid is not None

        await self.require_no_moderation(self.merchant, self.functionality.type)

        # Mapping dicts to dataclass
        data = self._map_merchant_data(
            addresses=self.addresses,
            bank=self.bank,
            organization=self.organization,
            persons=self.persons,
            username=self.username,
            fast_moderation=self.fast_moderation,
        )
        self._fill_missing_organization_names(data)

        # Getting or creating merchant without saving it yet
        if self.merchant is not None:
            new_merchant = False
            await CheckUpdateMerchantRolesAction().run()
            self.merchant.load_data()

            await self.ignore_ongoing_moderations(self.merchant, self.functionality.type)

            if self.merchant.status != MerchantStatus.DRAFT:  # PAYBACK-266, PAYBACK-786
                assert self.merchant.organization and data.organization
                if self.merchant.organization.inn != data.organization.inn:
                    self.logger.context_push(stored_inn=self.merchant.organization.inn,
                                             modified_inn=data.organization.inn)
                    self.logger.error('Inn modification is not allowed')
                    raise InnModificationError

            self.merchant.name = self.name
            self.merchant.data = data
            self.merchant.status = MerchantStatus.NEW
        else:
            new_merchant = True
            await CheckUIDMatchAction(uid=self.uid).run()

            self.merchant = Merchant(
                acquirer=AcquirerType(settings.DEFAULT_ACQUIRER),
                uid=self.uid,
                name=self.name,
                data=data,
            )

            self.merchant.load_parent()
            self.merchant.load_data()

        if self.merchant.acquirer == AcquirerType.KASSA and self.merchant.options.allowed_order_sources is None:
            self.merchant.options.allowed_order_sources = [OrderSource.SDK_API, OrderSource.SERVICE]
        assert self.merchant.data is not None
        self.merchant.data.registered = True

        self.logger.context_push(uid=self.merchant.uid, new_merchant=new_merchant)

        # Saving or creating merchant in database
        if new_merchant:
            merchant = await CreateMerchantEntityAction(self.merchant).run()
            merchant.oauth = []
            self.merchant = merchant

        assert self.merchant is not None

        self.logger.context_push(revision=self.merchant.revision)
        self.logger.info(f'Merchant {"created" if new_merchant else "updated"}.')

        if self.functionality.type == FunctionalityType.YANDEX_PAY:
            await self._init_yandex_pay_functionality()
        else:
            await self._init_payments_functionality()

        self.merchant = await self.storage.merchant.save(self.merchant)
        await self._load_parent()
        await self._load_oauth()
        await self._load_functionalities()
        self.merchant.load_parent()
        self.merchant.load_data()

        # Change log
        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=self.merchant.revision,
            operation=OperationKind.ADD_MERCHANT if new_merchant else OperationKind.EDIT_MERCHANT,
            arguments={
                'name': self.merchant.name,
                'data': json_value(self.merchant.data) if self.merchant.data else {},
                'client_id': self.merchant.client_id,
                'person_id': self.merchant.person_id,
                'submerchant_id': (
                    self.merchant.submerchant_id
                    if settings.TINKOFF_IGNORE_MERCHANT_CREATE_FAIL
                    else self.merchant.get_submerchant_id()
                ),
            }
        ))

        return self.merchant
