from typing import cast

from mail.payments.payments.core.actions.base.action import BaseAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.entities.enums import CallbackMessageType
from mail.payments.payments.core.entities.merchant import Merchant, PersonType
from mail.payments.payments.core.entities.not_fetched import NOT_FETCHED
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.interactions.balance.entities import Person
from mail.payments.payments.interactions.balance.exceptions import BalancePersonNotFound


class MerchantDataUpdateAction(APICallbackTaskMixin, BaseAction):
    def __init__(self, merchant: Merchant):
        super().__init__()
        self.merchant = merchant

    @staticmethod
    def _update_merchant_from_person(merchant: Merchant, person: Person) -> bool:
        """
        :return: True if need to send notifications else False
        """
        data = merchant.data

        assert data is not None
        assert data.addresses is not None
        assert data.bank is not None
        assert data.persons is not None
        assert data.organization is not None
        assert data.get_person(PersonType.CEO) is not None

        return BaseMerchantAction.fill_merchant_from_person(merchant, person)

    async def _send_notifications(self):
        async for service_merchant in self.storage.service_merchant.find(self.merchant.uid):
            service_id = service_merchant.service_id
            async for service_client in self.storage.service_client.find(service_id=service_id, with_service=True):
                service = cast(Service, service_client.service)
                service_client.service = NOT_FETCHED
                service.service_merchant = service_merchant
                service.service_client = service_client
                callback_message_type = CallbackMessageType.MERCHANT_REQUISITES_UPDATED
                await self.create_service_merchant_callback_task(service, callback_message_type)

    async def handle(self) -> Merchant:
        assert self.merchant is not None

        send_notifications = False
        if self.merchant.person_id is None:
            # для пререгистрированного мерчанта все None-ми измазано
            self.merchant.load_data()
            return self.merchant
        try:
            person = await self.clients.balance.get_person(self.merchant.person_id)
            assert person is not None
            send_notifications = self._update_merchant_from_person(self.merchant, person)
            self.merchant.load_data()
        except BalancePersonNotFound as e:
            with self.logger:
                self.logger.context_push(person_id=self.merchant.person_id)
                self.logger.error('Person not found in balance')
            raise e
        except KeyError as e:
            with self.logger:
                self.logger.context_push(missing_key=e.args[0])
                self.logger.error('Balance person missing fields. Falling back to data from db')
            self.merchant.load_data()

        if send_notifications:
            await self._send_notifications()

        return self.merchant
