from typing import Any, Dict, Iterable, Optional, Tuple

from sendr_utils import alist

from mail.payments.payments.core.actions.base.merchant import BaseDBAction, BaseMerchantAction
from mail.payments.payments.core.actions.merchant.check_uid_match import CheckUIDMatchAction
from mail.payments.payments.core.actions.merchant.create_entity import CreateMerchantEntityAction
from mail.payments.payments.core.actions.service_merchant.base import BaseServiceMerchantAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import (
    AcquirerType, MerchantDraftPolicy, MerchantRole, MerchantStatus, OperationKind, PersonType
)
from mail.payments.payments.core.entities.merchant import Merchant, MerchantOptions
from mail.payments.payments.core.entities.service import Service, ServiceMerchant
from mail.payments.payments.core.exceptions import MerchantCreateDeny
from mail.payments.payments.storage.exceptions import ServiceMerchantNotFound
from mail.payments.payments.utils.datetime import date_to_str_optional


class CheckUpdateRolesMerchantDraftAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.ADMIN,)

    async def handle(self) -> None:
        pass


class CoreMerchantDraftAction(BaseMerchantAction):
    transact = True
    allow_none = True
    skip_data = True
    skip_parent = True
    skip_moderation = True
    for_update = True
    draft_policy = MerchantDraftPolicy.MERCHANT_DRAFT_REQUIRED

    def __init__(self,
                 uid: int,
                 name: str,
                 addresses: Optional[Iterable[dict]] = None,
                 bank: Optional[dict] = None,
                 organization: Optional[dict] = None,
                 persons: Optional[Iterable[dict]] = None,
                 username: Optional[str] = None,
                 merchant: Optional[Merchant] = None,
                 options: Optional[MerchantOptions] = None,
                 check_service_merchant_present: bool = True,
                 acquirer: Optional[AcquirerType] = None,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.name: str = name
        self.addresses = addresses
        self.bank = bank
        self.organization = organization
        self.persons = persons
        self.username = username
        self.options = options
        self.check_service_merchant_present = check_service_merchant_present
        self.acquirer = acquirer

    @staticmethod
    def to_draft_data(addresses: Optional[Iterable[dict]] = None,
                      bank: Optional[dict] = None,
                      organization: Optional[dict] = None,
                      persons: Optional[Iterable[dict]] = None,
                      username: Optional[str] = None,
                      ) -> Dict:
        if persons is not None:
            for person in persons:
                if person.get('type') is not None:
                    assert isinstance(person['type'], PersonType)
                    person['type'] = person['type'].value
                if person.get('birth_date') is not None:
                    person['birth_date'] = date_to_str_optional(person['birth_date'])

        if organization is not None:
            organization['type'] = organization['type'].value

        return {
            'addresses': addresses,
            'bank': bank,
            'organization': organization,
            'persons': persons,
            'username': username,
            'registered': False,
        }

    @staticmethod
    def is_street(token: str) -> bool:
        street_types = (
            'аллея', 'ал.', 'бульвар', 'б-р', 'взвоз', 'взв.', 'въезд', 'взд.', 'дорога', 'дор.', 'заезд', 'ззд.',
            'километр', 'км', 'кольцо', 'к-цо', 'коса', 'линия', 'лн.', 'магистраль', 'мгстр.', 'набережная', 'наб.',
            'переезд', 'пер-д', 'переулок', 'пер.', 'площадка', 'пл-ка', 'площадь', 'пл.', 'проезд', 'пр-д', 'просек',
            'пр-к', 'просека', 'пр-ка', 'проселок', 'пр-лок', 'проспект', 'пр-кт', 'проулок', 'проул.', 'разъезд',
            'рзд.', 'ряд', 'сквер', 'с-р', 'спуск', 'с-к', 'съезд', 'сзд.', 'тракт', 'тупик', 'туп.', 'улица', 'ул.',
            'шоссе', 'ш.',
        )
        return token.lower().startswith(street_types)

    def get_address_dict_by_spark_suggest_str(self, spark_suggest_address: str) -> dict:
        address_tokens = spark_suggest_address.split(', ')
        if not address_tokens:
            return {}

        return {
            'zip': address_tokens[0],
            'city': next(filter(lambda token: token.startswith('г. '), address_tokens), None),
            'street': next(filter(lambda token: self.is_street(token), address_tokens), None),
            'home': next(filter(lambda token: token.startswith('д. '), address_tokens), None),
        }

    async def prefill_draft_data_by_inn(self, draft_data: dict) -> None:
        if draft_data.get('organization') is None:
            return

        inn = draft_data['organization'].get('inn')
        if inn is None:
            return

        spark_suggest_items = await self.clients.spark_suggest.get_hint(inn)
        if not spark_suggest_items:
            return

        for spark_suggest_item in spark_suggest_items:
            if spark_suggest_item.inn == inn:
                for field in ('name', 'full_name', 'ogrn'):
                    draft_data['organization'].setdefault(field, getattr(spark_suggest_item, field))

                spark_suggest_address = self.get_address_dict_by_spark_suggest_str(spark_suggest_item.address)
                draft_data['addresses'] = draft_data['addresses'] or []
                if not next(filter(lambda addr: addr.get('type') == 'legal', draft_data['addresses']), None):
                    draft_data['addresses'].append({'type': 'legal'})
                for address in draft_data['addresses']:
                    address.update({
                        k: v for k, v in spark_suggest_address.items()
                        if address['type'] == 'legal' and k not in address
                    })
                return

    async def handle(self) -> Merchant:
        self.logger.context_push(uid=self.uid, draft=True)

        draft_data = self.to_draft_data(addresses=self.addresses,
                                        bank=self.bank,
                                        organization=self.organization,
                                        persons=self.persons,
                                        username=self.username)

        await self.prefill_draft_data_by_inn(draft_data)

        if self.merchant is not None:
            await CheckUpdateRolesMerchantDraftAction().run()
            if self.check_service_merchant_present:
                # Если черновик был создан сервисом, не даем его редактировать
                if await alist(self.storage.service_merchant.find(self.merchant.uid, limit=1)):
                    raise MerchantCreateDeny

            self.merchant.name = self.name
            self.merchant.draft_data = draft_data
            self.merchant.status = MerchantStatus.DRAFT
            new_merchant = False
        else:
            assert self.uid is not None
            await CheckUIDMatchAction(uid=self.uid).run()
            self.merchant = Merchant(
                uid=self.uid,
                name=self.name,
                draft_data=draft_data,
                status=MerchantStatus.DRAFT,
            )
            new_merchant = True

        if self.options is not None:
            self.merchant.options = self.options
        if self.acquirer is not None:
            self.merchant.acquirer = self.acquirer

        # Saving or creating merchant draft in database
        if new_merchant:
            self.merchant = await CreateMerchantEntityAction(self.merchant).run()
        else:
            self.merchant = await self.storage.merchant.save(self.merchant)

        assert self.merchant is not None

        self.merchant.load_data()
        self.merchant.oauth = []

        self.logger.context_push(revision=self.merchant.revision)
        self.logger.info('Merchant draft created or updated')

        # Change log
        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=self.merchant.revision,
            operation=OperationKind.ADD_MERCHANT_DRAFT if new_merchant else OperationKind.EDIT_MERCHANT_DRAFT,
            arguments={
                'name': self.merchant.name,
                'draft_data': self.merchant.draft_data
            }
        ))

        return self.merchant


class CreateMerchantDraftAction(BaseDBAction):
    transact = True

    def __init__(self, **kwargs):
        super().__init__()
        self.kwargs = kwargs

    async def handle(self) -> Merchant:
        return await CoreMerchantDraftAction(**self.kwargs).run()


class CreateServiceMerchantDraftAction(BaseServiceMerchantAction):
    """Создание черновика через internal API сервисом
    """
    transact = True
    service: Service
    service_merchant: ServiceMerchant
    check_service_merchant_present = False

    def __init__(self,
                 uid: int,
                 service_tvm_id: int,
                 entity_id: str,
                 description: str,
                 autoenable: bool = False,
                 **kwargs: Any):
        super().__init__()
        self.service_tvm_id: int = service_tvm_id
        self.entity_id: str = entity_id
        self.description: str = description
        self.autoenable: bool = autoenable
        self.uid = uid
        self.kwargs = kwargs

    async def pre_handle(self) -> None:
        await super().pre_handle()
        self.service = await self.get_service(self.service_tvm_id)

        try:
            self.service_merchant = await self.storage.service_merchant.get(uid=self.uid,
                                                                            service_id=self.service.service_id,
                                                                            entity_id=self.entity_id,
                                                                            ignore_deleted=True,
                                                                            for_update=True)
        except ServiceMerchantNotFound:
            # Если черновик уже есть, и создан либо фронтом, либо другим сервисом, не даем редактировать
            if await self.storage.merchant.get_found_count(uid=self.uid) > 0:
                raise MerchantCreateDeny

            service_id = self.service.service_id
            assert service_id

            self.service_merchant = ServiceMerchant(uid=self.uid,
                                                    service_id=service_id,
                                                    entity_id=self.entity_id,
                                                    description=self.description)

        self.logger.context_push(service_id=self.service.service_id)

    async def handle(self) -> Tuple[ServiceMerchant, Merchant]:
        service_options = self.service.options

        merchant = await CoreMerchantDraftAction(
            uid=self.uid,
            check_service_merchant_present=False,
            options=MerchantOptions(
                offer_settings=service_options.offer_settings,
                allowed_order_sources=service_options.allowed_order_sources,
                allow_create_service_merchants=service_options.allow_create_service_merchants,
                hide_commission=service_options.hide_commission,
            ),
            acquirer=service_options.required_acquirer,
            **self.kwargs
        ).run()

        if self.autoenable:
            self.service_merchant.enabled = True

        if self.service_merchant.service_merchant_id:
            self.service_merchant = await self.storage.service_merchant.save(self.service_merchant)
        else:
            self.service_merchant = await self.storage.service_merchant.create(self.service_merchant)

        return self.service_merchant, merchant
