from typing import ClassVar
from uuid import uuid4

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.enums import FunctionalityType
from mail.payments.payments.core.entities.functionality import (
    MerchantFunctionality, PaymentsFunctionalityData, YandexPayFunctionalityData, YandexPayMerchantFunctionalityData,
    YandexPayPaymentGatewayFunctionalityData
)
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.exceptions import (
    MerchantContactIsEmptyError, YandexPayPartnerTypeChangedError, YandexPayPSPExternalIDChangedError,
    YandexPayPSPExternalIDEmptyError, YandexPaySchemaValidationError
)
from mail.payments.payments.interactions.yandex_pay_admin.entities import Contact
from mail.payments.payments.interactions.yandex_pay_admin.exceptions import (
    PartnerTypeChangedYaPayAdminError, PSPExternalIDChangedYaPayAdminError, PSPExternalIDEmptyYaPayAdminError,
    SchemaValidationYaPayAdminError
)


class PutPaymentsMerchantFunctionalityAction(BaseDBAction):
    transact = True
    functionality_type: ClassVar[FunctionalityType] = FunctionalityType.PAYMENTS

    def __init__(self, merchant: Merchant, data: PaymentsFunctionalityData):
        super().__init__()
        self.merchant = merchant
        self.data = data

    async def _create_new_functionality(self) -> MerchantFunctionality:
        return await self.storage.functionality.create(
            MerchantFunctionality(
                uid=self.merchant.uid,
                functionality_type=self.functionality_type,
                data=self.data,
            )
        )

    async def handle(self) -> MerchantFunctionality:
        try:
            functionality = await self.storage.functionality.get(
                self.merchant.uid, self.functionality_type, for_update=True,
            )
        except MerchantFunctionality.DoesNotExist:
            return await self._create_new_functionality()

        functionality.data = self.data
        return await self.storage.functionality.save(functionality)


class PutYandexPayMerchantFunctionalityAction(BaseDBAction):
    transact = True
    functionality_type: ClassVar[FunctionalityType] = FunctionalityType.YANDEX_PAY

    def __init__(self, merchant: Merchant, data: YandexPayFunctionalityData):
        super().__init__()
        self.merchant = merchant
        self.data = data

    async def _create_new_functionality(self) -> MerchantFunctionality:
        self.data.partner_id = uuid4()
        return await self.storage.functionality.create(
            MerchantFunctionality(
                uid=self.merchant.uid,
                functionality_type=self.functionality_type,
                data=self.data,
            )
        )

    def _get_merchant_name(self) -> str:
        if self.merchant.name:
            return self.merchant.name
        if self.merchant.organization is not None and self.merchant.organization.full_name:
            return self.merchant.organization.full_name
        return ''

    @classmethod
    def _update_functionality_data(
        cls,
        data: YandexPayFunctionalityData,
        previous_data: YandexPayFunctionalityData,
    ) -> YandexPayFunctionalityData:
        data.partner_id = previous_data.partner_id
        return data

    async def _put_functionality_to_yandex_pay_admin(self, functionality: MerchantFunctionality) -> None:
        if self.merchant.contact is None:
            raise MerchantContactIsEmptyError

        functionality_data = functionality.data
        assert (
            functionality.functionality_type == FunctionalityType.YANDEX_PAY
            and isinstance(functionality_data, YandexPayFunctionalityData)
        )
        partner_id = functionality_data.partner_id
        assert partner_id is not None
        name = self._get_merchant_name()

        contact = Contact(
            email=self.merchant.contact.email,
            phone=self.merchant.contact.phone,
            name=self.merchant.contact.name,
            surname=self.merchant.contact.surname,
            patronymic=self.merchant.contact.patronymic,
        )
        uid = self.merchant.uid

        try:
            if isinstance(functionality_data, YandexPayMerchantFunctionalityData):
                await self.clients.yandex_pay_admin.put_partner(
                    partner_id=partner_id,
                    name=name,
                    contact=contact,
                    partner_type=functionality_data.partner_type,
                    uid=uid,
                    merchant_gateway_id=functionality_data.merchant_gateway_id,
                    merchant_desired_gateway=functionality_data.merchant_desired_gateway,
                )
            elif isinstance(functionality_data, YandexPayPaymentGatewayFunctionalityData):
                await self.clients.yandex_pay_admin.put_partner(
                    partner_id=partner_id,
                    name=name,
                    contact=contact,
                    psp_external_id=functionality_data.gateway_id,
                    partner_type=functionality_data.partner_type,
                    payment_gateway_type=functionality_data.payment_gateway_type,
                    uid=uid,
                )
            else:
                raise RuntimeError(f'Unknown functionality partner_type {functionality_data.partner_type}')
        except PartnerTypeChangedYaPayAdminError:
            raise YandexPayPartnerTypeChangedError
        except PSPExternalIDChangedYaPayAdminError:
            raise YandexPayPSPExternalIDChangedError
        except SchemaValidationYaPayAdminError as exc:
            raise YandexPaySchemaValidationError(exc.params)
        except PSPExternalIDEmptyYaPayAdminError:
            raise YandexPayPSPExternalIDEmptyError

    async def handle(self) -> MerchantFunctionality:
        try:
            functionality = await self.storage.functionality.get(
                self.merchant.uid, self.functionality_type, for_update=True,
            )
        except MerchantFunctionality.DoesNotExist:
            functionality = await self._create_new_functionality()
        else:
            functionality.data = self._update_functionality_data(self.data, functionality.data)
            functionality = await self.storage.functionality.save(functionality)

        await self._put_functionality_to_yandex_pay_admin(functionality)
        return functionality


put_merchant_functionality = {
    FunctionalityType.PAYMENTS: PutPaymentsMerchantFunctionalityAction,
    FunctionalityType.YANDEX_PAY: PutYandexPayMerchantFunctionalityAction,
}
