from datetime import datetime
from typing import Any, AsyncIterable, Callable, Dict, List, Optional

from sendr_utils import alist

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.enums import (
    GraphType, GroupType, MerchantRole, OrderKind, PayStatus, RefundStatus
)
from mail.payments.payments.core.entities.merchant import AllPaymentsPoint, Merchant
from mail.payments.payments.core.exceptions import GraphTypeNotFoundError


class MerchantPaymentsGraphAction(BaseMerchantAction):
    skip_data = True
    skip_parent = True
    required_merchant_roles = (MerchantRole.VIEWER,)

    def __init__(self,
                 lower_dt: datetime,
                 upper_dt: datetime,
                 graph_type: GraphType,
                 group_by: GroupType = GroupType.DAY,
                 uid: Optional[int] = None,
                 pay_method: Optional[str] = None,
                 shop_id: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.lower_dt = lower_dt
        self.upper_dt = upper_dt
        self.graph_type = graph_type
        self.group_by = group_by
        self.pay_method = pay_method
        self.shop_id = shop_id

    async def handle(self) -> List[AllPaymentsPoint]:
        graph_type = self.graph_type

        graph_types: Dict[GraphType, Callable[..., AsyncIterable[AllPaymentsPoint]]] = {
            GraphType.ALL_PAYMENTS_COUNT: self.storage.order.payments_count,
            GraphType.PAID_COUNT: self.storage.order.payments_count,
            GraphType.REFUND_COUNT: self.storage.order.payments_count,
            GraphType.AVERAGE_BILL: self.storage.order.average_bill,
            GraphType.SUM_PAID_BILLS: self.storage.order.payments_sum,
            GraphType.REFUND_SUM: self.storage.order.payments_sum,
        }

        if graph_type not in graph_types:
            raise GraphTypeNotFoundError

        kwargs: Dict[str, Any] = {
            'lower_dt': self.lower_dt,
            'upper_dt': self.upper_dt,
            'group_by': self.group_by,
            'pay_method': self.pay_method,
            'shop_id': self.shop_id,
            'uid': self.uid,
            'iterator': True
        }

        if graph_type in [GraphType.PAID_COUNT, GraphType.SUM_PAID_BILLS, GraphType.AVERAGE_BILL]:
            kwargs['pay_status'] = PayStatus.PAID

        if graph_type in [GraphType.REFUND_COUNT, GraphType.REFUND_SUM]:
            kwargs['refund_status'] = RefundStatus.COMPLETED
            kwargs['order_kind'] = OrderKind.REFUND

        return await alist(graph_types[graph_type](**kwargs))
