from datetime import timedelta

from sendr_utils import utcnow

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.merchant_oauth import MerchantOAuth, OAuthToken
from mail.payments.payments.core.exceptions import OAuthCodeError, OAuthInvalidGrants
from mail.payments.payments.interactions.exceptions import OAuthClientError, OAuthInvalidGrantsClientError


class OAuthRefreshMerchantAction(BaseDBAction):
    def __init__(self, merchant_oauth: MerchantOAuth):
        super().__init__()
        self.merchant_oauth = merchant_oauth

    async def handle(self) -> MerchantOAuth:
        refresh_token = self.merchant_oauth.decrypted_refresh_token
        # merchant_id=str(self.merchant_oauth.uid) для совместимости логов. Раньше ключом oauth был merchant_id
        self.logger.context_push(merchant_id=str(self.merchant_oauth.uid), uid=self.merchant_oauth.uid)

        try:
            oauth_token: OAuthToken = await self.clients.oauth.refresh_token(refresh_token)
        except OAuthInvalidGrantsClientError as exc:
            self.logger.context_push(error=exc.params)
            self.logger.error('Invalid grants error due refresh_token')
            raise OAuthInvalidGrants
        except OAuthClientError as exc:
            self.logger.context_push(error=exc.params)
            self.logger.error('OAuth refresh_token error')
            raise OAuthCodeError

        self.merchant_oauth.decrypted_access_token = oauth_token.access_token
        self.merchant_oauth.decrypted_refresh_token = oauth_token.refresh_token
        self.merchant_oauth.expires = utcnow() + timedelta(seconds=oauth_token.expires_in)

        return await self.storage.merchant_oauth.save(self.merchant_oauth)
