import asyncio
from datetime import datetime
from typing import Any, Dict, List, Optional, TypedDict

from aiopg.sa import SAConnection

from sendr_interactions import exceptions as interaction_errors
from sendr_utils import alist, json_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.merchant.check_uid_match import CheckUIDMatchAction
from mail.payments.payments.core.actions.merchant.create_entity import CreateMerchantEntityAction
from mail.payments.payments.core.actions.merchant.functionality import put_merchant_functionality
from mail.payments.payments.core.entities.category import Category
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import AcquirerType, FunctionalityType, OperationKind, PersonType
from mail.payments.payments.core.entities.functionality import FunctionalityData
from mail.payments.payments.core.entities.merchant import (
    Merchant, MerchantData, MerchantOptions, OrganizationData, PersonData
)
from mail.payments.payments.core.entities.merchant_preregistration import (
    MerchantPreregistration, MerchantPreregistrationData, PreregisterData
)
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.exceptions import (
    CategoryNotFoundError, ConflictingAcquirerType, InnIsEmptyError, MerchantExistsPreregisterError,
    MerchantInactivePreregisterError, MerchantPreregistrationNotFoundError, ServiceNotFoundError
)
from mail.payments.payments.interactions.balance.entities import Person
from mail.payments.payments.interactions.search_wizard.entities import Fio
from mail.payments.payments.storage.exceptions import MerchantPreregistrationNotFound
from mail.payments.payments.utils.helpers import is_entrepreneur_by_inn


class ServiceRecordErrorParam(TypedDict):
    service_id: int
    required_acquirer: str


class CategoryRecordErrorParam(TypedDict):
    category_id: int
    required_acquirer: str


class ChooseAcquirerErrorParams(TypedDict):
    services: List[ServiceRecordErrorParam]
    categories: List[CategoryRecordErrorParam]


class MerchantPreregistrationAcquirerChooser:
    """
    Вспомогательный класс, хранящий в себе логику определения типа эквайера для продавца.

    PAYBACK-627
    """

    def __init__(self, requires_online: bool, services: List[Service], categories: List[Category]):
        """
        @param requires_online: внешний признак (задается пользователем) необходимости онлайн платежей для продавца
        @param services: сервисы, которые хочет подключить продавец

        @raise ConflictingAcquirerType: сервисы и категории задают противоречивые требования на тип эквайера
        """
        self._requires_online = requires_online
        self._services = services
        self._categories = categories

    def _get_error_params(self) -> ChooseAcquirerErrorParams:
        """
        Вспомогательная информация для сообщения ошибки выбора эквайера:
        перечисляем идентификаторы сервисов / категорий с явными требованиями на эквайер.
        """

        error_params: ChooseAcquirerErrorParams = {
            'services': [],
            'categories': [],
        }

        service: Service
        for service in self._services:
            assert service.service_id is not None
            if service.options.required_acquirer is not None:
                service_record: ServiceRecordErrorParam = {
                    'service_id': service.service_id,
                    'required_acquirer': service.options.required_acquirer.value,
                }
                error_params['services'].append(service_record)

        category: Category
        for category in self._categories:
            assert category.category_id is not None
            if category.required_acquirer is not None:
                category_record: CategoryRecordErrorParam = {
                    'category_id': category.category_id,
                    'required_acquirer': category.required_acquirer.value,
                }
                error_params['categories'].append(category_record)

        return error_params

    def _get_services_acquirer(self) -> Optional[AcquirerType]:
        """
        Тип эквайера, задаваемый списком сервисов.
        При выборе типа эквайера учитываем только те сервисы,
        у которых в опциях явно прописан требуемый тип эквайера -
        если тип не прописан явно (None значение), то считаем,
        что такой сервис не налагает ограничений на тип эквайера.
        """

        acquirers: List[AcquirerType] = list(set(
            service.options.required_acquirer
            for service in self._services
            if service.options.require_online and service.options.required_acquirer is not None
        ))

        if len(acquirers) > 1:
            raise ConflictingAcquirerType(params=self._get_error_params())  # type: ignore

        if len(acquirers) == 1:
            return acquirers[0]

        # у каждого сервиса нет явного требования по эквайеру
        return None

    def _get_categories_acquirer(self) -> Optional[AcquirerType]:
        """
        Тип эквайера, задаваемый списком категорий.
        При выборе типа эквайера учитываем только те категории,
        у которых в опциях явно прописан требуемый тип эквайера -
        если тип не прописан явно (None значение), то считаем,
        что такая категория не налагает ограничений на тип эквайера.
        """

        acquirers: List[AcquirerType] = list(set(
            category.required_acquirer
            for category in self._categories
            if category.required_acquirer is not None
        ))

        if len(acquirers) > 1:
            raise ConflictingAcquirerType(params=self._get_error_params())  # type: ignore

        if len(acquirers) == 1:
            return acquirers[0]

        return None

    def get_requires_online(self) -> bool:
        """Определяем необходимость онлайн платажей"""
        return self._requires_online or any((
            service.options.require_online for service in self._services
        ))

    def _combine_acquirers(
        self,
        acquirer_one: Optional[AcquirerType],
        acquirer_two: Optional[AcquirerType],
    ) -> Optional[AcquirerType]:
        if acquirer_one and acquirer_two and acquirer_one != acquirer_two:
            raise ConflictingAcquirerType(params=self._get_error_params())  # type: ignore
        return acquirer_one or acquirer_two

    def _get_common_acquirer(self) -> AcquirerType:
        service_acquirer = self._get_services_acquirer()
        categories_acquirer = self._get_categories_acquirer()
        common_acquirer = self._combine_acquirers(service_acquirer, categories_acquirer)
        if common_acquirer:
            return common_acquirer
        else:
            return AcquirerType(settings.DEFAULT_ACQUIRER)

    def get_acquirer(self) -> Optional[AcquirerType]:
        """
        Определяем общий тип требуемого эквайера.
        """
        if not self.get_requires_online():
            return None
        return self._get_common_acquirer()

    def get_can_skip_registration(self) -> bool:
        return any([service.options.can_skip_registration for service in self._services])


class PreregisterMerchantAction(BaseMerchantAction):
    """
    Создаем практически пустого мерчанта: делим процесс регистрации на этапы.

    После пререгистрации можно "проводить" через нас оффлайн заказы (де факто это просто учет метаданных).

    Создание пререгистрированного мерчанта - действие одноразовое - если в базе есть сущность с uid-ом,
    то кидаем ошибку.

    See:
        PAYBACK-626
    """
    transact = False
    allow_none = True
    skip_parent = True
    skip_data = True
    skip_moderation = True
    skip_oauth = True
    skip_preregistration = True
    for_update = True

    def __init__(
        self,
        inn: Optional[str],
        services: List[int],
        require_online: bool,
        categories: List[int],
        functionality: FunctionalityData,
        uid: Optional[int] = None,
        merchant: Optional[Merchant] = None,
        spark_id: Optional[int] = None,
        contact: Optional[Dict[str, Any]] = None,
    ):
        super().__init__(uid=uid, merchant=merchant)
        self.inn = inn
        self.services = services
        self.require_online = require_online
        self.categories = categories
        self.spark_id = spark_id
        self.contact = contact
        self.functionality = functionality
        self.is_merchant_new = True

    async def _create_merchant_registration_chooser(self) -> MerchantPreregistrationAcquirerChooser:
        required_services = set(self.services)
        services: List[Service] = await alist(self.storage.service.find(service_ids=required_services))
        found_services = set(s.service_id for s in services)
        not_found_services = required_services - found_services
        if not_found_services:
            raise ServiceNotFoundError(service_id=next(iter(not_found_services)))

        required_categories = set(self.categories)
        categories: List[Category] = await alist(self.storage.category.find(category_ids=required_categories))
        found_categories = set(c.category_id for c in categories)
        not_found_categories = required_categories - found_categories
        if not_found_categories:
            raise CategoryNotFoundError(category_id=next(iter(not_found_categories)))

        return MerchantPreregistrationAcquirerChooser(
            requires_online=self.require_online,
            services=services,
            categories=categories,
        )

    async def _log_to_change_log(self, operation: OperationKind) -> None:
        assert self.merchant

        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=self.merchant.revision,
            operation=operation,
            arguments={
                'name': self.merchant.name,
                'data': json_value(self.merchant.data) if self.merchant.data else {},
            }
        ))

    async def _get_person_from_balance(self) -> Optional[Person]:
        assert self.uid is not None
        assert self.merchant is not None

        if not self.inn:
            # INN не обязателен для yandex pay YANDEXPAY-2374
            return None

        if is_entrepreneur_by_inn(self.inn):  # disable prefill entrepreneur merchant from balance PAYBACK-797
            return None

        try:
            client = await self.clients.balance.find_client(self.uid,
                                                            total_timeout=settings.BALANCE_PREFILL_TIMEOUT)
            if not client:
                return None
            persons = await self.clients.balance.get_client_persons(client.client_id,
                                                                    total_timeout=settings.BALANCE_PREFILL_TIMEOUT)
        except interaction_errors.BaseInteractionError as e:

            self.logger.warning(f'Fail to fetch persosns from balance for prefill merchant: {e}')
            return None
        matched_persons = sorted(
            filter(lambda person: person.inn == self.inn, persons),
            key=lambda person: person.date.timestamp() if person.date is not None else 0,
            reverse=True
        )
        return matched_persons[0] if matched_persons else None

    async def _prefill_merchant_from_spark(self, person_date: Optional[datetime]) -> None:
        assert self.merchant is not None
        assert self.merchant.data is not None

        if not self.inn:
            # INN не обязателен для yandex pay YANDEXPAY-2374
            return

        try:
            spark_data = await self.clients.spark.get_info(self.inn, self.spark_id)
        except interaction_errors.BaseInteractionError as e:
            self.logger.warning(f'Fail to fetch spark data for prefill merchant: {e}')
            return

        if not spark_data.active:
            raise MerchantInactivePreregisterError()

        fio = Fio(
            first_name='',
            middle_name=None,
            last_name=spark_data.leader_name or '',
        )
        try:
            fio = await self.clients.search_wizard.split_fio(fio.last_name) or fio
        except interaction_errors.BaseInteractionError as e:
            self.logger.warning(f'Fail to parse FIO through search wizard: {e}')

        stored_ts = (
            person_date
            if self.is_merchant_new or (person_date is not None and person_date > self.merchant.updated)
            else self.merchant.updated
        )

        data = self.merchant.data
        data.addresses = spark_data.get_patched_addresses(data.addresses, stored_ts)
        data.organization = spark_data.get_patched_organization(data.organization, stored_ts)
        data.persons = spark_data.get_patched_persons(data.persons,
                                                      fio.first_name,
                                                      fio.last_name,
                                                      fio.middle_name,
                                                      stored_ts)

    async def _prefill_bank_from_cbrf(self) -> None:
        assert self.merchant is not None

        if self.merchant.data is None or self.merchant.data.bank is None or not self.merchant.data.bank.bik:
            return

        bank = self.merchant.data.bank
        try:
            bank_requisites = await self.clients.refs.cbrf_bank(bank.bik)
        except interaction_errors.BaseInteractionError as e:
            self.logger.warning(f'Fail to fetch bank from cbrf for prefill merchant: {e}')
            return

        bank.name = bank_requisites.name_full
        bank.correspondent_account = bank_requisites.corr

    async def _create_or_update_merchant_registration(self) -> MerchantPreregistration:
        assert self.uid is not None
        registration_entity = MerchantPreregistration(uid=self.uid, data=MerchantPreregistrationData())
        raw_preregister_data = PreregisterData(
            inn=self.inn,
            services=self.services,
            categories=self.categories,
            require_online=self.require_online,
        )
        registration_entity.data.raw_preregister_data = raw_preregister_data

        registration, is_created = await self.storage.merchant_preregistration.get_or_create(
            registration_entity,
            lookup_fields=('uid',),
            for_update=True,
        )

        if not is_created:
            # "грязные" данные пререгистрации всегда сохраняем и обновляем
            registration.data.raw_preregister_data = raw_preregister_data
            registration = await self.storage.merchant_preregistration.save(registration)

        return registration

    async def _create_or_update_preregistered_merchant(self, registration: MerchantPreregistration) -> None:
        assert self.uid is not None
        acquirer_chooser = await self._create_merchant_registration_chooser()

        preregister_data = PreregisterData(
            inn=self.inn,
            services=self.services,
            categories=self.categories,
            require_online=acquirer_chooser.get_requires_online(),
        )
        registration.data.preregister_data = preregister_data
        await self.storage.merchant_preregistration.save(registration)

        data = MerchantData(
            organization=OrganizationData(inn=self.inn),
            persons=[
                PersonData(
                    type=PersonType.CONTACT,
                    name=self.contact['name'],
                    email=self.contact['email'],
                    phone=self.contact['phone'],
                    surname=self.contact['surname'],
                    patronymic=self.contact.get('patronymic'),
                    birth_date=self.contact.get('birth_date'),
                )
            ] if self.contact else [],
            registered=False,
        )
        options = MerchantOptions(can_skip_registration=acquirer_chooser.get_can_skip_registration())

        if self.is_merchant_new:
            self.merchant = Merchant(uid=self.uid, data=data, options=options)
        else:
            assert self.merchant is not None
            if self.merchant.organization and self.merchant.organization.inn != self.inn:
                self.merchant.person_id = None

            await self.ignore_ongoing_moderations(self.merchant, self.functionality.type)

            self.merchant.data = data
            self.merchant.options = options

        self.merchant.acquirer = acquirer_chooser.get_acquirer()

        if self.is_merchant_new or (self.merchant.data is not None and not self.merchant.data.registered):
            person = await self._get_person_from_balance()
            person_date = None
            if person:
                BaseMerchantAction.fill_merchant_from_person(self.merchant, person)
                person_date = person.date
            await asyncio.gather(self._prefill_merchant_from_spark(person_date), self._prefill_bank_from_cbrf())

        if self.is_merchant_new:
            self.merchant = await CreateMerchantEntityAction(self.merchant).run()
            await self._log_to_change_log(OperationKind.ADD_MERCHANT_PREREGISTRATION)
        else:
            self.merchant = await self.storage.merchant.save(self.merchant)
            await self._log_to_change_log(OperationKind.EDIT_MERCHANT_PREREGISTRATION)

        assert self.merchant is not None

        self.merchant.load_data()

        await put_merchant_functionality[self.functionality.type](
            merchant=self.merchant,
            data=self.functionality,
        ).run()
        await self._load_functionalities()

    def _ensure_merchant_not_registered_completely(self) -> None:
        if self.merchant is not None and self.merchant.registered:
            raise MerchantExistsPreregisterError(params=dict(uid=self.merchant.uid))

    def _check_inn(self) -> None:
        # YANDEXPAY-2374
        if self.functionality.type != FunctionalityType.YANDEX_PAY:
            if self.inn is None:
                raise InnIsEmptyError

    async def pre_handle(self) -> None:
        await super().pre_handle()
        if self.merchant is not None:
            self.is_merchant_new = False
            self.merchant.load_data()

    async def handle(self) -> Merchant:
        """
        Создание или обновление регистрационной записи должно происходить независимо от
        успешности попытки создать мерчанта.
        """
        assert self.uid is not None
        self.logger.context_push(uid=self.uid)
        self._ensure_merchant_not_registered_completely()
        await self.require_no_moderation(self.merchant, self.functionality.type)
        await CheckUIDMatchAction(uid=self.uid).run()

        connection: SAConnection = self.storage.conn

        self._check_inn()

        async with connection.begin_nested():
            registration = await self._create_or_update_merchant_registration()

        async with connection.begin_nested():
            await self._create_or_update_preregistered_merchant(registration=registration)

        assert self.merchant is not None
        self.logger.context_push(revision=self.merchant.revision)
        self.logger.info(f'Merchant preregistration {"created" if self.is_merchant_new else "updated"}.')
        self.merchant.preregistration = await self.storage.merchant_preregistration.get(uid=self.merchant.uid)
        return self.merchant


class GetMerchantPreregistrationAction(BaseDBAction):
    transact = False

    def __init__(self, uid: int):
        super().__init__()
        self.uid = uid

    async def handle(self) -> MerchantPreregistration:
        try:
            return await self.storage.merchant_preregistration.get(uid=self.uid)
        except MerchantPreregistrationNotFound:
            raise MerchantPreregistrationNotFoundError(uid=self.uid)
