from datetime import datetime
from decimal import Decimal
from typing import List, Optional

from sendr_utils import alist

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.enums import AcquirerType
from mail.payments.payments.core.entities.merchant import MerchantStat

DEFAULT_COMMISSION = Decimal(settings.PAYMENTS_COMMISSION)


class MerchantsOrdersStatsAction(BaseDBAction):
    def __init__(self,
                 closed_from: Optional[datetime] = None,
                 closed_to: Optional[datetime] = None,
                 service_id: Optional[int] = None,
                 acquirer: Optional[AcquirerType] = None):
        super().__init__()
        self.closed_from = closed_from
        self.closed_to = closed_to
        self.service_id = service_id
        self.acquirer = acquirer

    async def handle(self) -> List[MerchantStat]:
        return await alist(
            self.storage.merchant.batch_orders_stats(
                closed_from=self.closed_from,
                closed_to=self.closed_to,
                acquirer=self.acquirer,
                service_id=self.service_id,
                default_commission=DEFAULT_COMMISSION,
            )
        )


class MerchantOrdersStatsAction(BaseMerchantAction):
    skip_parent = True
    skip_data = True
    skip_moderation = True
    skip_oauth = True

    def __init__(self,
                 uid: int,
                 date_from: Optional[datetime] = None,
                 date_to: Optional[datetime] = None):
        super().__init__(uid=uid)
        self.date_from = date_from
        self.date_to = date_to

    async def handle(self) -> MerchantStat:
        assert self.uid
        return await self.storage.merchant.orders_stats(
            uid=self.uid,
            date_from=self.date_from,
            date_to=self.date_to,
            default_commission=DEFAULT_COMMISSION,
        )
