from typing import Dict, Optional, Tuple

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.customer_subscription import CustomerSubscription
from mail.payments.payments.core.entities.enums import CallbackMessageType, OrderKind
from mail.payments.payments.core.entities.merchant import APICallbackParams, Merchant
from mail.payments.payments.core.entities.moderation import Moderation
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service, ServiceClient, ServiceMerchant
from mail.payments.payments.core.entities.task import Task, TaskType

CallbackParams = Tuple[Optional[str], Optional[int], Dict, Optional[APICallbackParams]]


class APICallbackTaskMixin(BaseDBAction):
    """Methods to create tasks which will call certain API endpoint with given message."""

    def _make_params(self, merchant: Optional[Merchant], service: Optional[Service]) -> CallbackParams:
        callback_params = None
        tvm_id = None

        if merchant is not None:
            callback_url = merchant.api_callback_url
            message_params = {'uid': merchant.uid}
            callback_params = merchant.api_callback_params
        elif service is not None:
            assert (isinstance(service.service_client, ServiceClient)
                    and isinstance(service.service_merchant, ServiceMerchant)
                    and service.service_merchant.service_merchant_id)

            callback_url = service.service_client.api_callback_url
            message_params = {'service_merchant_id': service.service_merchant.service_merchant_id}
            tvm_id = service.service_client.tvm_id
        else:
            raise RuntimeError('Merchant or service is required')

        return callback_url, tvm_id, message_params, callback_params

    async def create_merchant_moderation_updated_task(self,
                                                      moderation: Moderation,
                                                      *,
                                                      merchant: Optional[Merchant] = None,
                                                      service: Optional[Service] = None) -> Optional[Task]:
        callback_url, tvm_id, message_params, callback_params = self._make_params(merchant, service)
        if not callback_url:
            return None

        task_params = dict(
            tvm_id=tvm_id,
            callback_message_type=CallbackMessageType.MERCHANT_MODERATION_UPDATED,
            callback_url=callback_url,
            callback_params=callback_params,
            message={
                'approved': moderation.approved,
                **message_params,
            },
        )

        task = await self.storage.task.create(Task(task_type=TaskType.API_CALLBACK, params=task_params))

        with self.logger as logger:
            logger.context_push(uid=moderation.uid)
            if service:
                logger.context_push(
                    service_id=service.service_id,
                    service_client_id=service.service_client.service_client_id,  # type: ignore
                    service_merchant_id=service.service_merchant.service_merchant_id  # type: ignore
                )
            logger.info('Callback scheduled on merchant moderation status update.')
        return task

    async def create_service_merchant_callback_task(self,
                                                    service: Service,
                                                    callback_message_type: CallbackMessageType) -> Optional[Task]:
        callback_url = service.service_client.api_callback_url  # type: ignore
        if not callback_url:
            return None

        task_params = dict(
            tvm_id=service.service_client.tvm_id,  # type: ignore
            callback_message_type=callback_message_type,
            callback_url=callback_url,
            message={'service_merchant_id': service.service_merchant.service_merchant_id},  # type: ignore
        )

        task = await self.storage.task.create(Task(task_type=TaskType.API_CALLBACK, params=task_params))

        with self.logger as logger:
            logger.context_push(
                service_id=service.service_id,
                uid=service.service_merchant.uid,  # type: ignore
                service_client_id=service.service_client.service_client_id,  # type: ignore
                service_merchant_id=service.service_merchant.service_merchant_id  # type: ignore
            )
            logger.info(callback_message_type.value + ' сallback scheduled.')
        return task

    async def create_order_status_task(self, order: Order, *,
                                       merchant: Optional[Merchant] = None,
                                       service: Optional[Service] = None) -> Optional[Task]:
        """Создать задачу на уведомление продавца или сервиса о статусе заказа."""
        assert order.pay_status is not None and order.updated is not None

        callback_url, tvm_id, message_params, callback_params = self._make_params(merchant, service)
        if not callback_url:
            return None

        task_params = dict(
            tvm_id=tvm_id,
            callback_message_type=CallbackMessageType.ORDER_STATUS_UPDATED,
            callback_url=callback_url,
            callback_params=callback_params,
            message={
                'order_id': order.order_id,
                'meta': order.data.meta,
                'new_status': order.pay_status.value,
                'updated': order.updated.isoformat(),
                **message_params,
            },
        )
        task = await self.storage.task.create(Task(task_type=TaskType.API_CALLBACK, params=task_params))

        with self.logger as logger:
            logger.context_push(
                uid=order.uid,
                order_id=order.order_id,
                order_pay_status=order.pay_status.value
            )
            if service:
                logger.context_push(
                    service_id=service.service_id,
                    service_client_id=service.service_client.service_client_id,  # type: ignore
                    service_merchant_id=service.service_merchant.service_merchant_id  # type: ignore
                )
            logger.info('Callback scheduled on order status update')
        return task

    async def create_refund_status_task(self,
                                        refund: Order,
                                        *,
                                        merchant: Optional[Merchant] = None,
                                        service: Optional[Service] = None,
                                        ) -> Optional[Task]:
        """Создать задачу на уведомление продавца или сервиса о статусе возврата."""
        assert refund.kind == OrderKind.REFUND and refund.refund_status is not None and refund.updated

        callback_url, tvm_id, message_params, callback_params = self._make_params(merchant, service)
        if not callback_url:
            return None

        task_params = dict(
            callback_message_type=CallbackMessageType.REFUND_STATUS_UPDATED,
            callback_url=callback_url,
            callback_params=callback_params,
            tvm_id=tvm_id,
            message={
                **message_params,
                'meta': refund.data.meta,
                'refund_id': refund.order_id,
                'order_id': refund.original_order_id,
                'new_status': refund.refund_status.value,
                'updated': refund.updated.isoformat(),
            }
        )
        task = await self.storage.task.create(Task(task_type=TaskType.API_CALLBACK, params=task_params))

        with self.logger:
            self.logger.context_push(
                **message_params,
                refund_id=refund.order_id,
                refund_refund_status=refund.refund_status.value,
            )
            self.logger.info('Refund status callback scheduled')

        return task

    async def create_service_merchant_updated_task(self, service: Service) -> Optional[Task]:
        """Create task to notify service about changes in ServiceMerchant relation."""
        callback_url, tvm_id, message_params, callback_params = self._make_params(None, service)
        if not callback_url:
            return None

        task_params = dict(
            callback_message_type=CallbackMessageType.SERVICE_MERCHANT_UPDATED,
            callback_url=callback_url,
            callback_params=callback_params,
            tvm_id=tvm_id,
            message={
                'enabled': service.service_merchant.enabled,  # type: ignore
                'entity_id': service.service_merchant.entity_id,  # type: ignore
                'updated': service.service_merchant.updated.isoformat(),  # type: ignore
                **message_params
            }
        )
        task = await self.storage.task.create(Task(task_type=TaskType.API_CALLBACK, params=task_params))

        with self.logger:
            self.logger.context_push(
                service_id=service.service_id,
                uid=service.service_merchant.uid,  # type: ignore
                service_client_id=service.service_client.service_client_id,  # type: ignore
                service_merchant_id=service.service_merchant.service_merchant_id,  # type: ignore
                enabled=service.service_merchant.enabled,  # type: ignore
            )
            self.logger.info('Service callback scheduled on service-merchant relation update')
        return task

    async def create_customer_subscription_prolongated_service_task(self,
                                                                    customer_subscription: CustomerSubscription,
                                                                    service: Service) -> Optional[Task]:
        callback_url, tvm_id, message_params, callback_params = self._make_params(None, service)
        if not customer_subscription.service_merchant_id or not callback_url:
            return None
        assert customer_subscription.updated is not None

        task_params = dict(
            tvm_id=tvm_id,
            callback_message_type=CallbackMessageType.CUSTOMER_SUBSCRIPTION_PROLONGATED,
            callback_url=callback_url,
            callback_params=callback_params,
            message={
                'order_id': customer_subscription.order_id,
                'customer_subscription_id': customer_subscription.customer_subscription_id,
                'updated': customer_subscription.updated.isoformat(),
                **message_params
            },
        )
        task = await self.storage.task.create(Task(task_type=TaskType.API_CALLBACK, params=task_params))

        with self.logger:
            self.logger.context_push(
                uid=customer_subscription.uid,
                order_id=customer_subscription.order_id,
                customer_subscription_id=customer_subscription.customer_subscription_id,
                service_id=service.service_id,
                service_client_id=service.service_client.service_client_id,  # type: ignore
                service_merchant_id=service.service_merchant.service_merchant_id,  # type: ignore
            )
            self.logger.info('Service callback scheduled on customer_subscription prolongated')

        return task

    async def create_customer_subscription_stopped_service_task(self,
                                                                customer_subscription: CustomerSubscription,
                                                                service: Service) -> Optional[Task]:
        callback_url, tvm_id, message_params, callback_params = self._make_params(None, service)
        if not customer_subscription.service_merchant_id or not callback_url:
            return None
        assert customer_subscription.updated is not None

        task_params = dict(
            tvm_id=tvm_id,
            callback_url=callback_url,
            callback_message_type=CallbackMessageType.CUSTOMER_SUBSCRIPTION_STOPPED,
            callback_params=callback_params,
            message={
                'order_id': customer_subscription.order_id,
                'customer_subscription_id': customer_subscription.customer_subscription_id,
                'updated': customer_subscription.updated.isoformat(),
                **message_params
            }
        )
        task = await self.storage.task.create(Task(task_type=TaskType.API_CALLBACK, params=task_params))

        with self.logger:
            self.logger.context_push(
                uid=customer_subscription.uid,
                order_id=customer_subscription.order_id,
                customer_subscription_id=customer_subscription.customer_subscription_id,
                service_id=service.service_id,
                service_client_id=service.service_client.service_client_id,  # type: ignore
                service_merchant_id=service.service_merchant.service_merchant_id,  # type: ignore
            )
            self.logger.info('Service callback scheduled on customer_subscription stopped')

        return task
