from typing import AsyncIterable, Optional

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.functionality import FunctionalityType
from mail.payments.payments.core.entities.merchant import Merchant, ModerationData
from mail.payments.payments.core.entities.moderation import Moderation, ModerationType
from mail.payments.payments.core.exceptions import CoreActionDenyError


class MerchantModerationMixin(BaseDBAction):
    @staticmethod
    def _effective_uid(merchant: Merchant) -> int:
        return merchant.uid if not merchant.parent_uid else merchant.parent_uid

    async def get_revision_moderation(
        self,
        uid: int,
        revision: int,
        functionality_type: FunctionalityType,
    ) -> Optional[Moderation]:
        """
        Returns merchant moderation for (uid, revision) or None if not found.
        Silently checks that there's only one such moderation.
        """
        moderations = [
            moderation
            async for moderation in self.storage.moderation.find(
                moderation_type=ModerationType.MERCHANT,
                functionality_type=functionality_type,
                uid=uid,
                revision=revision,
                ignore=False,
            )
        ]
        if not moderations:
            return None
        elif len(moderations) > 1:
            with self.logger:
                self.logger.context_push(
                    uid=uid,
                    revision=revision,
                    moderation_ids=[m.moderation_id for m in moderations],
                )
                self.logger.error('Merchant has more than 1 moderation for 1 revision.')
        return moderations[0]

    async def get_effective_moderation(
        self,
        merchant: Merchant,
        functionality_type: FunctionalityType,
    ) -> Optional[Moderation]:
        return await self.storage.moderation.get_effective(
            uid=self._effective_uid(merchant),
            moderation_type=ModerationType.MERCHANT,
            functionality_type=functionality_type
        )

    async def get_ongoing_moderations(
        self, merchant: Merchant, functionality_type: FunctionalityType, for_update: bool = False,
    ) -> AsyncIterable[Moderation]:
        async for moderation in self.storage.moderation.find(
            uid=self._effective_uid(merchant),
            moderation_type=ModerationType.MERCHANT,
            functionality_type=functionality_type,
            has_approved=False,
            ignore=False,
            for_update=for_update,
        ):
            yield moderation

    async def has_ongoing_moderations(
        self,
        merchant: Merchant,
        functionality_type: FunctionalityType,
    ) -> bool:
        if (
            merchant.moderations is not None
            and merchant.moderations.get(functionality_type) is not None
        ):
            moderation = merchant.moderations[functionality_type]
            if moderation.has_ongoing is not None:
                return moderation.has_ongoing
        ongoing = [m async for m in self.get_ongoing_moderations(merchant, functionality_type=functionality_type)]
        return bool(ongoing)

    async def ignore_ongoing_moderations(self, merchant: Merchant, functionality_type: FunctionalityType) -> None:
        async for moderation in self.get_ongoing_moderations(
            merchant, functionality_type=functionality_type, for_update=True
        ):
            moderation.ignore = True
            await self.storage.moderation.save(moderation)
            with self.logger:
                self.logger.context_push(moderation_id=moderation.moderation_id)
                self.logger.info('Ongoing moderation ignored')

    async def approved_effective_moderation(
        self, merchant: Merchant, functionality_type: FunctionalityType,
    ) -> Optional[bool]:
        if merchant.moderations is not None and merchant.moderations.get(functionality_type) is not None:
            return merchant.moderations[functionality_type].approved
        moderation = await self.get_effective_moderation(merchant, functionality_type=functionality_type)
        return moderation is not None and moderation.approved

    async def require_moderation(self, merchant: Optional[Merchant], functionality_type: FunctionalityType) -> None:
        # везде где требуется наличие положительной модерации на продавца нужно, чтобы продавец был незаблокирован
        if (
            merchant is None
            or merchant.blocked
            or not await self.approved_effective_moderation(merchant, functionality_type=functionality_type)
        ):
            raise CoreActionDenyError

    async def require_no_moderation(self, merchant: Optional[Merchant], functionality_type: FunctionalityType) -> None:
        if merchant is not None and await self.approved_effective_moderation(merchant, functionality_type):
            raise CoreActionDenyError

    async def get_moderation_data(self, merchant: Merchant, functionality_type: FunctionalityType) -> ModerationData:
        effective = await self.get_effective_moderation(merchant=merchant, functionality_type=functionality_type)
        has_ongoing = await self.has_ongoing_moderations(merchant=merchant, functionality_type=functionality_type)
        return ModerationData(
            approved=effective.approved if effective is not None else False,
            reasons=effective.reasons if effective is not None else [],
            has_moderation=has_ongoing or effective is not None,
            has_ongoing=has_ongoing,
            updated=effective.updated if effective is not None else None,
        )
