from typing import Optional, cast

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.entities.enums import (
    CallbackMessageType, FunctionalityType, MerchantRole, ModerationType, PaymentsTestCase, TaskType
)
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.moderation import Moderation
from mail.payments.payments.core.entities.not_fetched import NOT_FETCHED
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.entities.subscription import Subscription
from mail.payments.payments.core.entities.task import Task
from mail.payments.payments.core.exceptions import (
    MerchantCannotScheduleModerationForChildError, MerchantCannotScheduleModerationForPreregisterError
)


class ScheduleMerchantModerationAction(APICallbackTaskMixin, BaseMerchantAction):
    transact = True
    skip_data = False
    skip_parent = False
    for_update = True

    required_merchant_roles = (MerchantRole.ADMIN,)

    def __init__(self,
                 functionality_type: FunctionalityType,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 skip_moderation_task: bool = False,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.skip_moderation_task: bool = skip_moderation_task
        self.functionality_type = functionality_type

    async def _send_notifications(self):
        async for service_merchant in self.storage.service_merchant.find(self.merchant.uid):
            service_id = service_merchant.service_id
            async for service_client in self.storage.service_client.find(service_id=service_id, with_service=True):
                service = cast(Service, service_client.service)
                service_client.service = NOT_FETCHED
                service.service_merchant = service_merchant
                service.service_client = service_client
                callback_message_type = CallbackMessageType.MERCHANT_MODERATION_STARTED
                await self.create_service_merchant_callback_task(service, callback_message_type)

    async def handle(self) -> Moderation:
        self.logger.context_push(functionality_type=self.functionality_type)
        assert self.merchant

        if any((
            self.merchant.addresses is None,
            self.merchant.bank is None,
            self.merchant.organization is None,
            self.merchant.acquirer is None,
        )):
            # нельзя модерировать пререгистрированного мерчанта
            raise MerchantCannotScheduleModerationForPreregisterError

        # Checking that merchant has no parents
        if self.merchant.parent_uid:
            raise MerchantCannotScheduleModerationForChildError

        await self.require_no_moderation(self.merchant, functionality_type=self.functionality_type)
        # Checking if we already have merchant moderation with same (uid, revision)
        moderation = await self.get_revision_moderation(
            self.merchant.uid, self.merchant.revision, self.functionality_type
        )
        if moderation:
            self.logger.context_push(moderation_id=moderation.moderation_id)
            self.logger.info('Merchant moderation with same revision already exists')
            return moderation
        # Creating task and moderation in database
        moderation = await self.storage.moderation.create(Moderation(
            uid=self.merchant.uid,
            revision=self.merchant.revision,
            moderation_type=ModerationType.MERCHANT,
            functionality_type=self.functionality_type
        ))
        if not self.skip_moderation_task:
            task = await self.storage.task.create(Task(
                task_type=TaskType.START_MODERATION,
                params=dict(merchant_uid=self.merchant.uid, functionality_type=self.functionality_type.value),
            ))
            self.logger.context_push(
                moderation_id=moderation.moderation_id,
                task_id=task.task_id,
            )
            self.logger.info('Merchant moderation scheduled')

            await self._send_notifications()

        return moderation


class ScheduleOrderModerationAction(BaseDBAction):
    transact = True

    def __init__(self, order: Order):
        super().__init__()
        self.order: Order = order

    async def handle(self) -> Moderation:
        if self.order.test is not None:
            return await self.storage.moderation.create(Moderation(
                uid=self.order.uid,
                entity_id=self.order.order_id,
                moderation_type=ModerationType.ORDER,
                revision=self.order.revision,
                approved=False if self.order.test == PaymentsTestCase.TEST_MODERATION_FAILED else True
            ))

        moderation = await self.storage.moderation.create(Moderation(
            uid=self.order.uid,
            entity_id=self.order.order_id,
            moderation_type=ModerationType.ORDER,
            revision=self.order.revision,
        ))
        assert moderation.moderation_id is not None
        await self.storage.task.create(Task(
            task_type=TaskType.START_ORDER_MODERATION,
            params=dict(moderation_id=moderation.moderation_id),
        ))
        self.logger.context_push(moderation_id=moderation.moderation_id)
        self.logger.info('Moderation is scheduled')
        return moderation


class ScheduleSubscriptionModerationAction(BaseDBAction):
    transact = True

    def __init__(self, subscription: Subscription):
        super().__init__()
        self.subscription: Subscription = subscription

    async def handle(self) -> Moderation:
        moderation = await self.storage.moderation.create(Moderation(
            uid=self.subscription.uid,
            entity_id=self.subscription.subscription_id,
            moderation_type=ModerationType.SUBSCRIPTION,
            revision=self.subscription.revision,
        ))

        assert moderation.moderation_id is not None
        await self.storage.task.create(Task(
            task_type=TaskType.START_SUBSCRIPTION_MODERATION,
            params=dict(moderation_id=moderation.moderation_id),
        ))

        self.logger.context_push(moderation_id=moderation.moderation_id)
        self.logger.info('Moderation is scheduled')

        return moderation
