import asyncio
import io
from datetime import date
from typing import Tuple

from xhtml2pdf.pisa import CreatePDF

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.enums import AcquirerType, MerchantRole
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.exceptions import CoreDataError
from mail.payments.payments.utils.jinja import env


class GetOfferAction(BaseMerchantAction):
    skip_moderation = True
    required_merchant_roles = (MerchantRole.ADMIN,)

    def get_headers(self):
        return {
            'Content-Type': 'application/pdf',
            'Content-Disposition': 'attachment; filename="offer.pdf"',
        }

    def create_pdf(self, rendered: str) -> bytes:
        buffer = io.BytesIO()
        CreatePDF(rendered, dest=buffer)
        value = buffer.getvalue()
        buffer.close()

        return value

    def render_pdf(self, merchant: Merchant) -> bytes:
        template_name: str
        if merchant.acquirer == AcquirerType.TINKOFF:
            template_name = 'offer-tinkoff.html'
        elif merchant.acquirer == AcquirerType.KASSA:
            template_name = 'offer-kassa.html'
        else:
            raise CoreDataError('Acquirer is invalid')

        offer_settings = merchant.options.offer_settings
        if offer_settings.slug is not None:
            assert offer_settings.pdf_template
            template = env.from_string(offer_settings.pdf_template)
        else:
            template = env.get_template(template_name)

        return self.create_pdf(
            template.render(today=date.today(), merchant=merchant)
        )

    async def handle(self) -> Tuple[dict, bytes]:
        headers = self.get_headers()
        loop = asyncio.get_event_loop()
        data = await loop.run_in_executor(None, lambda: self.render_pdf(self.merchant))  # type: ignore
        return headers, data
