from typing import Optional

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import OperationKind, PayStatus
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.storage.exceptions import ServiceNotFound


class AbandonTerminateOrderAction(APICallbackTaskMixin, BaseOrderAction, BaseMerchantAction):
    transact = True
    skip_data = True
    skip_moderation = True

    def __init__(self, order: Order, uid: Optional[int] = None, merchant: Optional[Merchant] = None):
        super().__init__(uid=uid, merchant=merchant)
        self.order = order

    async def handle(self) -> Order:
        assert self.merchant

        # update status
        self.order.pay_status = PayStatus.ABANDONED
        self.order = await self.storage.order.save(self.order)
        assert self.order.order_id is not None
        await SendToHistoryOrderAction(uid=self.order.uid, order_id=self.order.order_id).run_async()

        if self.order.service_client_id is not None and self.order.service_merchant_id is not None:
            try:
                service = await self.storage.service.get_by_related(
                    service_client_id=self.order.service_client_id,
                    service_merchant_id=self.order.service_merchant_id,
                )
                await self.create_order_status_task(order=self.order, service=service)
            except ServiceNotFound:
                pass

        # schedule callback
        await self.create_order_status_task(order=self.order, merchant=self.merchant)

        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=self.order.revision,  # type: ignore
            operation=OperationKind.UPDATE_ORDER,
            arguments={
                'pay_status': PayStatus.ABANDONED.value,
                'order_id': self.order.order_id
            }
        ))

        self.order.items = await self._fetch_items(self.order)
        await self.pushers.log.push(self._create_log_instance(self.merchant, self.order, is_update=True))

        return self.order
