from typing import Optional

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.enums import MerchantRole
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.exceptions import OrderAlreadyPaidError, OrderNotFoundError
from mail.payments.payments.storage.exceptions import OrderNotFound


class CoreActivateOrderAction(BaseOrderAction, BaseMerchantAction):
    transact = True
    skip_data = True
    skip_parent = True

    def __init__(self,
                 order_id: int,
                 active: Optional[bool] = None,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 skip_add_crypto: bool = False,
                 with_customer_subscription: bool = False,
                 select_customer_subscription: Optional[bool] = False,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.order_id: int = order_id
        self.active: Optional[bool] = active
        self.skip_add_crypto = skip_add_crypto
        self.with_customer_subscription: bool = with_customer_subscription
        self.select_customer_subscription = select_customer_subscription

    async def _get_order(self) -> Order:
        assert self.merchant
        return await self.storage.order.get(
            uid=self.merchant.uid,
            order_id=self.order_id,
            for_update=True,
            with_customer_subscription=self.with_customer_subscription,
            select_customer_subscription=self.select_customer_subscription,
        )

    async def handle(self) -> Order:
        assert self.uid

        try:
            order = await self._get_order()
        except OrderNotFound:
            raise OrderNotFoundError(uid=self.uid, order_id=self.order_id)

        self.logger.context_push(order_id=order.order_id, revision=order.revision, active=order.active)

        # Checking if there's anything to update
        if self.active is not None and self.active != order.active:
            if not self.active and order.is_already_paid:
                raise OrderAlreadyPaidError
            order.active = self.active
        else:
            self.logger.info('Nothing to update')
            return order

        # Saving order
        order = await self.storage.order.save(order)
        assert order.order_id is not None
        await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()

        # Logging
        await self.storage.change_log.create(ChangeLog(
            uid=self.uid,
            revision=order.revision,  # type: ignore
            operation=OperationKind.UPDATE_ORDER,
            arguments={'active': order.active, 'order_id': order.order_id},
        ))
        self.logger.info('Order update: active')

        if not self.skip_add_crypto:
            order.add_crypto(settings.CRYPTO_V1_F1_PREFIX, self.crypto)

        return order


class ActivateOrderAction(CoreActivateOrderAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)


class ActivateOrderServiceMerchantAction(AuthServiceMerchantMixin, CoreActivateOrderAction):
    def __init__(self,
                 service_tvm_id: int,
                 service_merchant_id: int,
                 order_id: int,
                 active: Optional[bool] = None,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 ):
        self.service_tvm_id: int = service_tvm_id
        self.service_merchant_id: int = service_merchant_id

        super().__init__(
            order_id=order_id,
            active=active,
            uid=uid,
            merchant=merchant,
        )

    async def _get_order(self) -> Order:
        assert self.merchant

        return await self.storage.order.get(
            uid=self.merchant.uid,
            order_id=self.order_id,
            service_merchant_id=self.service_merchant_id,
            for_update=True
        )
