from collections import Counter
from decimal import Decimal, DecimalException
from typing import Iterable, List, Optional, Type, Union

from sendr_utils import enum_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.enums import OrderSource
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.log import OrderCreatedLog, OrderUpdatedLog
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.exceptions import (
    CoreFieldError, ItemsInvalidMarkupSumError, OrderHasDuplicateItemEntriesError, OrderPriceExceeds100kRUBError
)


class BaseOrderAction(BaseDBAction):
    service: Optional[Service] = None

    @staticmethod
    def _items_price(items: Iterable[Item]) -> Decimal:
        return sum([
            item.amount * item.price  # type: ignore
            for item in items
        ], Decimal(0))

    async def _fetch_items(self, order: Order) -> List[Item]:
        assert order.order_id is not None

        return [
            item
            async for item in self.storage.item.get_for_order(
                uid=order.uid,
                order_id=order.order_id,
            )
        ]

    @staticmethod
    def _validate_item_markup(item: Item) -> None:
        if (markup := item.markup) is None:
            return
        if 'card' not in markup:
            raise CoreFieldError(fields={'card': 'Is required in markup'})
        markup_sum = Decimal('0')
        for field in markup:
            if field not in settings.ITEM_MARKUP_AVAILABLE_FIELDS:
                raise CoreFieldError(fields={field: 'Unexpected field in markup'})
            try:
                markup_field = Decimal(markup[field])
            except (DecimalException, TypeError):
                raise CoreFieldError(fields={field: 'A decimal value expected'})
            if markup_field != markup_field.quantize(Decimal('1.00')):
                raise CoreFieldError(fields={field: 'A decimal value with precision 2 expected'})
            if markup_field.is_signed():  # https://st.yandex-team.ru/PAYBACK-901
                raise CoreFieldError(fields={field: 'A non-negative value expected'})
            markup_sum += markup_field
        if markup_sum != item.amount * item.price:  # type: ignore
            raise ItemsInvalidMarkupSumError

    def _validate_order_items(self, items: Iterable[Item], merchant: Merchant) -> None:
        def item_product_keys(items):
            for item in items:
                assert item.product is not None
                yield item.product.key_out

        c: Counter = Counter(item_product_keys(items))
        duplicates = [elem for elem, cnt in c.most_common() if cnt > 1]
        if duplicates:
            self.logger.info('Validate order error: duplicates found')
            raise OrderHasDuplicateItemEntriesError

        total_price = float(round(self._items_price(items), 2))

        limit = merchant.options.order_max_total_price if merchant.options.order_max_total_price is not None \
            else settings.ORDER_MAX_TOTAL_PRICE

        if total_price > limit:
            self.logger.info(f'Validate order error: total price exceeds {limit}')
            raise OrderPriceExceeds100kRUBError(limit)

        for item in items:
            self._validate_item_markup(item)

    def _create_log_instance(self,
                             merchant: Merchant,
                             order: Order,
                             is_update: bool = False,
                             extra: Optional[dict] = None) -> Union[OrderUpdatedLog, OrderCreatedLog]:
        assert order.order_id and order.items
        cls: Union[Type[OrderUpdatedLog], Type[OrderCreatedLog]] = OrderUpdatedLog if is_update else OrderCreatedLog

        return cls(
            merchant_name=merchant.name,
            merchant_uid=merchant.uid,
            merchant_acquirer=order.get_acquirer(merchant.acquirer),
            order_id=order.order_id,
            kind=order.kind.value,
            status=enum_value(order.pay_status),
            price=order.log_price,
            items=[item.dump() for item in order.items],
            customer_subscription_id=order.customer_subscription_id,
            customer_uid=order.customer_uid,
            merchant_oauth_mode=enum_value(order.merchant_oauth_mode),
            sdk_api_created=order.created_by_source == OrderSource.SDK_API,
            sdk_api_pay=order.pay_by_source == OrderSource.SDK_API,
            created_by_source=order.created_by_source,
            pay_by_source=order.pay_by_source,
            **(extra or {})
        )
