from typing import Any, Optional

from sendr_utils import utcnow

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.get import CoreGetOrderAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import (
    MerchantRole, OperationKind, OrderKind, PayStatus, TransactionStatus
)
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.exceptions import (
    OrderAbandonedError, OrderAlreadyPaidError, OrderArchivedError, OrderCancelledError,
    OrderHaveUnfinishedTransactions
)


class CoreCancelOrderAction(APICallbackTaskMixin, BaseOrderAction, BaseMerchantAction):
    """
    Отменить заказ, при условии что на нём нет активной (незавершенной) транзакции.
    Заказ нельзя будет оплатить после этого действия.
    """
    transact = True
    skip_data = True

    def __init__(self,
                 order_id: int,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 select_customer_subscription: Optional[bool] = False):
        super().__init__(uid=uid, merchant=merchant)
        self.order_id = order_id
        self.select_customer_subscription = select_customer_subscription

    async def _check_can_cancel(self, order: Order) -> None:
        assert order.order_id and order.pay_status
        self.logger.context_push(pay_status=order.pay_status.value)

        if order.is_already_paid:
            raise OrderAlreadyPaidError
        elif order.pay_status == PayStatus.CANCELLED:
            raise OrderCancelledError
        elif order.pay_status == PayStatus.ABANDONED:
            raise OrderAbandonedError
        elif not order.active:
            raise OrderArchivedError

        tx = await self.storage.transaction.get_last_by_order(uid=order.uid, order_id=order.order_id, raise_=False)
        if tx is not None:
            self.logger.context_push(tx_id=tx.tx_id, transaction_status=tx.status.value)
            if not tx.finished:
                raise OrderHaveUnfinishedTransactions
            elif tx.status == TransactionStatus.CLEARED:
                raise OrderAlreadyPaidError

    async def handle(self) -> Order:
        assert self.uid and self.merchant
        self.logger.context_push(order_id=self.order_id, uid=self.uid)

        order = await CoreGetOrderAction(
            uid=self.uid,
            order_id=self.order_id,
            kind=OrderKind.PAY,
            for_update=True,
            select_customer_subscription=self.select_customer_subscription
        ).run()

        await self._check_can_cancel(order)

        order.pay_status = PayStatus.CANCELLED
        order.closed = utcnow()
        items = order.items
        shop = order.shop

        order = await self.storage.order.save(order)
        order.items = items
        order.shop = shop

        assert order.revision and order.pay_status
        self.logger.context_push(new_pay_status=order.pay_status)

        # schedule callback
        if order.service_client_id is not None and order.service_merchant_id is not None:
            service = await self.storage.service.get_by_related(
                service_client_id=order.service_client_id,
                service_merchant_id=order.service_merchant_id,
            )
            await self.create_order_status_task(order=order, service=service)
        await self.create_order_status_task(order=order, merchant=self.merchant)

        await self.storage.change_log.create(ChangeLog(
            uid=order.uid,
            revision=order.revision,
            operation=OperationKind.UPDATE_ORDER,
            arguments={
                'pay_status': order.pay_status.value,
                'order_id': order.order_id
            }
        ))
        await self.pushers.log.push(
            self._create_log_instance(self.merchant, order, is_update=True)
        )

        self.logger.info('Order canceled manually.')
        return order


class CancelOrderAction(BaseDBAction):
    transact = True
    required_merchant_roles = (MerchantRole.OPERATOR,)

    def __init__(self, **kwargs: Any):
        super().__init__()
        self.kwargs = kwargs

    async def handle(self) -> Order:
        return await CoreCancelOrderAction(**self.kwargs).run()


class CancelOrderServiceMerchantAction(AuthServiceMerchantMixin, BaseDBAction):
    transact = True

    def __init__(self,
                 service_merchant_id: Optional[int] = None,
                 service_tvm_id: Optional[int] = None,
                 service: Optional[Service] = None,
                 **kwargs: Any,
                 ):
        self.service_merchant_id: Optional[int] = service_merchant_id
        self.service_tvm_id: Optional[int] = service_tvm_id
        self.service: Optional[Service] = service
        self.kwargs = kwargs
        super().__init__()

    async def handle(self) -> Order:
        assert self.uid
        return await CoreCancelOrderAction(uid=self.uid, **self.kwargs).run()
