from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.interactions.trust import ClearPaymentInTrustAction
from mail.payments.payments.core.actions.merchant.get_acquirer import GetAcquirerMerchantAction
from mail.payments.payments.core.actions.order.get import GetOrderAction
from mail.payments.payments.core.entities.enums import OrderKind
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.transaction import Transaction
from mail.payments.payments.core.exceptions import OrderNotFoundError, OrdersAmountExceed, TransactionNotFoundError
from mail.payments.payments.storage.exceptions import OrderNotFound, TransactionNotFound


class ClearOrderAction(BaseDBAction):
    transact = True

    def __init__(self, transaction: Transaction, order: Order):
        super().__init__()
        self.transaction: Transaction = transaction
        self.order: Order = order

    async def handle(self) -> None:
        if self.order.parent_order_id:
            multi_order: Order = await GetOrderAction(
                uid=self.order.uid,
                order_id=self.order.parent_order_id,
                kind=OrderKind.MULTI,
                skip_add_crypto=True,
                for_update=True
            ).run()

            if multi_order.multi_amount_exceed:
                raise OrdersAmountExceed
            multi_order.data.multi_issued += 1
            await self.storage.order.save(multi_order)

        acquirer = await GetAcquirerMerchantAction(uid=self.order.uid).run()
        acquirer = self.order.get_acquirer(acquirer)

        assert self.transaction.trust_purchase_token
        await ClearPaymentInTrustAction(
            purchase_token=self.transaction.trust_purchase_token,
            order=self.order,
            acquirer=acquirer
        ).run()


class ClearByIdsOrderAction(BaseDBAction):
    action_name = 'clear_by_ids_order_action'

    def __init__(self, uid: int, order_id: int, tx_id: int):
        super().__init__()
        self.uid: int = uid
        self.order_id: int = order_id
        self.tx_id: int = tx_id

    async def handle(self) -> None:
        try:
            order = await self.storage.order.get(uid=self.uid, order_id=self.order_id)
        except OrderNotFound:
            raise OrderNotFoundError

        try:
            transaction = await self.storage.transaction.get(self.uid, self.tx_id)
        except TransactionNotFound:
            raise TransactionNotFoundError

        await ClearOrderAction(transaction=transaction, order=order).run()
