from decimal import Decimal
from typing import Dict, List, Optional, Tuple, Type, cast

from sendr_core import BaseAction
from sendr_utils import enum_value

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.interactions.trust import UnholdPaymentInTrustAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.clear import ClearByIdsOrderAction
from mail.payments.payments.core.actions.order.get_trust_env import GetOrderTrustEnvAction
from mail.payments.payments.core.actions.order.resize import ResizeOrderAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.actions.update_transaction import UpdateTransactionAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.enums import FunctionalityType, MerchantRole, OrderKind, PayStatus, TrustEnv
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.transaction import Transaction
from mail.payments.payments.core.exceptions import (
    CoreActionDenyError, CoreDataError, ItemsInvalidAmountDataError, ItemsInvalidDataError,
    ItemsInvalidTotalPriceDataError, OrderCannotClearOrUnholdRefundError, OrderHasAutoclearOnError, OrderNotFoundError,
    OrderPayStatusMustBeHeldOrInModerationError, TransactionNotFoundError
)
from mail.payments.payments.storage.exceptions import OrderNotFound, TransactionNotFound


class ClearUnholdOrderMixin(BaseMerchantAction):
    transact = True
    skip_data = True

    def __init__(self,
                 order_id: int,
                 operation: str,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 items: Optional[List[dict]] = None
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.order_id = order_id
        self.operation = operation
        self.items = items

    async def _fetch_entities(self, uid: int, order_id: int) -> Tuple[Order, Transaction]:
        assert self.merchant

        try:
            await self.storage.order.get(uid, order_id)  # проверка на то, что заказ есть, и это не подписка
            transaction = await self.storage.transaction.get_last_by_order(uid, order_id, for_update=True, raise_=True)
            assert transaction is not None
            transaction = await UpdateTransactionAction(
                merchant=self.merchant,
                transaction=transaction,
            ).run()
            order = await self.storage.order.get(uid, order_id, for_update=True)
        except OrderNotFound:
            raise OrderNotFoundError(uid=uid, order_id=order_id)
        except TransactionNotFound:
            raise TransactionNotFoundError(uid=uid, order_id=order_id)
        assert transaction

        return order, transaction

    async def _check_order(self, order: Order) -> None:
        assert self.merchant
        trust_env = await GetOrderTrustEnvAction(order=order).run()

        if trust_env != TrustEnv.SANDBOX:
            await self.require_moderation(self.merchant, functionality_type=FunctionalityType.PAYMENTS)
        if order.get_acquirer(self.merchant.acquirer) is None:
            raise CoreActionDenyError
        if order.kind != OrderKind.PAY:
            self.logger.info(f'Cannot clear or unhold {enum_value(order.kind)}')
            raise OrderCannotClearOrUnholdRefundError
        if order.autoclear:
            self.logger.info('Order has autoclear on')
            raise OrderHasAutoclearOnError
        if order.pay_status not in (PayStatus.HELD, PayStatus.IN_MODERATION):
            self.logger.info('Order pay_status is not "held" or "in_moderation"')
            raise OrderPayStatusMustBeHeldOrInModerationError(params={'pay_status': enum_value(order.pay_status)})

    async def _fetch_validated_items(self, order: Order, items: List[dict]) -> List[Item]:
        validated_items = []

        uid_and_order_id_list: List[Tuple[int, int]] = [(order.uid, cast(int, order.order_id))]

        db_items_dict: Dict[int, Item] = {
            cast(int, item.product_id): item async for item in self.storage.item.get_for_orders(
                uid_and_order_id_list, iterator=True
            )
        }

        for param in items:
            validated_item = Item(
                uid=order.uid,
                order_id=order.order_id,
                amount=param['amount'],
                product_id=param['product_id'],
                new_price=param['price']
            )

            try:
                db_item = db_items_dict[param['product_id']]
            except KeyError:
                raise ItemsInvalidDataError

            assert validated_item.price is not None and validated_item.total_price is not None
            assert db_item.price is not None and db_item.total_price is not None
            if validated_item.amount > db_item.amount:
                raise ItemsInvalidAmountDataError
            if validated_item.total_price > db_item.total_price:
                raise ItemsInvalidTotalPriceDataError

            validated_items.append(validated_item)

        return validated_items

    async def _fetch_order_tx_and_items(self) -> Tuple[Order, Transaction, List[Item]]:
        assert self.merchant
        order, transaction = await self._fetch_entities(self.merchant.uid, self.order_id)

        assert order.pay_status \
               and transaction.status \
               and transaction.trust_purchase_token

        self.logger.context_push(
            order_id=order.order_id,
            tx_id=transaction.tx_id,
            operation=self.operation,
            pay_status=order.pay_status.value,
            transaction_status=transaction.status.value,
        )

        try:
            await self._check_order(order)
        except CoreDataError:
            await self.storage.commit()
            raise

        items = []
        if self.operation == 'clear' and self.items is not None:
            items = await self._fetch_validated_items(order=order, items=self.items)
        return order, transaction, items


class CoreClearUnholdOrderAction(ClearUnholdOrderMixin):
    transact = True
    skip_data = True
    action_name = 'core_clear_unhold_order_action'

    @classmethod
    def deserialize_kwargs(cls: Type[BaseAction], init_kwargs: dict) -> dict:
        items = init_kwargs.get('items')
        if items is None:
            return init_kwargs
        for item in items:
            item['price'] = Decimal(item['price'])
            item['amount'] = Decimal(item['amount'])
        return init_kwargs

    @classmethod
    def serialize_kwargs(cls: Type[BaseAction], init_kwargs: dict) -> dict:
        items = init_kwargs.get('items')
        if items is None:
            return init_kwargs
        for item in items:
            item['price'] = str(item['price'])
            item['amount'] = str(item['amount'])
        return init_kwargs

    async def handle(self) -> None:
        self.logger.context_push(order_id=self.order_id, uid=self.uid, operation=self.operation)

        try:
            order, transaction, items = await self._fetch_order_tx_and_items()
        except OrderPayStatusMustBeHeldOrInModerationError as e:
            self.logger.context_push(exception=e)
            self.logger.warning('pay_status invalid due operation')
            return

        assert self.merchant is not None
        assert order.order_id is not None
        assert transaction.tx_id is not None

        if self.operation == 'clear':
            if order.pay_status == PayStatus.HELD:
                order.pay_status = PayStatus.IN_PROGRESS
                if items:
                    await ResizeOrderAction(
                        transaction=transaction,
                        order=order,
                        items=items,
                    ).run()
                await ClearByIdsOrderAction(
                    uid=order.uid,
                    order_id=order.order_id,
                    tx_id=transaction.tx_id
                ).run_async()
                self.logger.info('Order clear requested in trust')
            else:
                order.autoclear = True
                self.logger.info('Cannot clear while order in moderation. Switching it to autoclear instead')
        elif self.operation == 'unhold':
            assert transaction.trust_purchase_token is not None
            await UnholdPaymentInTrustAction(
                purchase_token=transaction.trust_purchase_token,
                order=order,
                acquirer=order.get_acquirer(self.merchant.acquirer),  # type: ignore
            ).run()
            order.pay_status = PayStatus.IN_CANCEL
            self.logger.info('Order unhold requested in trust')
        else:
            raise ValueError('Operation must be "clear" or "unhold"')

        self.logger.context_push(new_pay_status=order.pay_status)

        transaction.poll = True
        transaction = await self.storage.transaction.save(transaction)
        assert transaction.revision
        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=transaction.revision,
            operation=OperationKind.UPDATE_TX,
            arguments={'poll': True},
        ))
        self.logger.info('Transaction poll enabled')

        order = await self.storage.order.save(order)
        assert order.order_id is not None and order.revision and order.pay_status
        await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()
        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=order.revision,
            operation=OperationKind.UPDATE_ORDER,
            arguments={
                'pay_status': order.pay_status.value,
                'order_id': order.order_id,
            },
        ))
        self.logger.info('Order updated: pay_status')


class ScheduleClearUnholdOrderAction(BaseDBAction):
    """Clears or unholds money. Instead of clearing money of in_moderation order, marks it for autoclear."""
    required_merchant_roles = (MerchantRole.OPERATOR,)
    transact = True

    def __init__(self,
                 order_id: int,
                 operation: str,
                 uid: Optional[int] = None
                 ):
        super().__init__()
        self.order_id = order_id
        self.operation = operation
        self.uid = uid

    async def handle(self) -> None:
        # all the necessary validation will be done in CoreClearUnholdOrderAction
        await CoreScheduleClearUnholdOrderAction(
            order_id=self.order_id,
            operation=self.operation,
            uid=self.uid
        ).run()


class ScheduleClearUnholdServiceMerchantOrderAction(AuthServiceMerchantMixin, ClearUnholdOrderMixin):
    def __init__(self,
                 service_merchant_id: int,
                 service_tvm_id: int,
                 order_id: int,
                 operation: str,
                 items: Optional[List[dict]] = None,
                 ):
        super().__init__(order_id=order_id, operation=operation, items=items)
        self.service_merchant_id = service_merchant_id
        self.service_tvm_id = service_tvm_id

    async def handle(self) -> None:
        # validation of input order and resize params
        await self._fetch_order_tx_and_items()
        await CoreClearUnholdOrderAction(
            order_id=self.order_id,
            operation=self.operation,
            uid=self.uid,
            items=self.items
        ).run_async()


class CoreScheduleClearUnholdOrderAction(ClearUnholdOrderMixin):
    async def handle(self) -> None:
        # validation of input order and resize params
        await self._fetch_order_tx_and_items()
        await CoreClearUnholdOrderAction(
            order_id=self.order_id,
            operation=self.operation,
            uid=self.uid,
            items=self.items
        ).run_async()
