from typing import List, Optional, Type

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.create_or_update import (
    CreateOrUpdateOrderAction, CreateOrUpdateOrderServiceMerchantAction
)
from mail.payments.payments.core.actions.order.get import GetOrderAction
from mail.payments.payments.core.entities.enums import MerchantRole, OrderKind
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.exceptions import OrderArchivedError, OrderNotFoundError, OrdersAmountExceed
from mail.payments.payments.storage.exceptions import OrderNotFound
from mail.payments.payments.storage.mappers.order.order import FindOrderParams
from mail.payments.payments.utils.helpers import create_csv_writer


class CoreCreateOrderFromMultiOrderAction(BaseDBAction):
    transact = True

    def __init__(self,
                 order_id: int,
                 uid: int,
                 create_order_action_class: Type[BaseDBAction],
                 create_order_extra: Optional[dict] = None):
        super().__init__()
        self.uid = uid
        self.order_id = order_id
        self.create_order_action_class = create_order_action_class
        self.create_order_extra: dict = create_order_extra or {}

    def _create_order_action_kwargs(self, multi_order: Order) -> dict:
        return dict(
            caption=multi_order.caption,
            items=multi_order.items,
            autoclear=multi_order.autoclear,
            description=multi_order.description,
            user_email=None,
            customer_uid=None,
            user_description=multi_order.user_description,
            return_url=multi_order.return_url,
            paymethod_id=multi_order.paymethod_id,
            test=multi_order.test,
            parent_order_id=multi_order.order_id,
            customer_subscription_id=multi_order.customer_subscription_id,
            kind=OrderKind.PAY,
            uid=self.uid,
            **self.create_order_extra,
        )

    async def handle(self) -> Order:
        assert self.uid

        try:
            multi_order: Order = await GetOrderAction(
                uid=self.uid,
                order_id=self.order_id,
                kind=OrderKind.MULTI,
                for_update=True
            ).run()
        except OrderNotFound:
            raise OrderNotFoundError

        if multi_order.multi_amount_exceed:
            raise OrdersAmountExceed

        if not multi_order.active:
            raise OrderArchivedError

        kwargs = self._create_order_action_kwargs(multi_order)
        CreateOrderAction = self.create_order_action_class
        return await CreateOrderAction(**kwargs).run()  # type: ignore


class CreateOrderFromMultiOrderAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)

    def __init__(self, order_id: int, uid: int):
        super().__init__()
        self.uid = uid
        self.order_id = order_id

    async def handle(self) -> Order:
        return await CoreCreateOrderFromMultiOrderAction(
            uid=self.uid,
            order_id=self.order_id,
            create_order_action_class=CreateOrUpdateOrderAction
        ).run()


class CreateOrderFromMultiOrderServiceMerchantAction(AuthServiceMerchantMixin, BaseDBAction):
    def __init__(self,
                 order_id: int,
                 service_merchant_id: Optional[int] = None,
                 service_tvm_id: Optional[int] = None,
                 service: Optional[Service] = None):
        super().__init__()
        self.order_id = order_id
        self.service_merchant_id: Optional[int] = service_merchant_id
        self.service_tvm_id: Optional[int] = service_tvm_id
        self.service: Optional[Service] = service

    async def handle(self) -> Order:
        assert self.uid

        create_order_extra = {
            'service_merchant_id': self.service_merchant_id,
            'service_tvm_id': self.service_tvm_id,
            'service': self.service
        }

        return await CoreCreateOrderFromMultiOrderAction(
            uid=self.uid,
            order_id=self.order_id,
            create_order_action_class=CreateOrUpdateOrderServiceMerchantAction,
            create_order_extra=create_order_extra,
        ).run()


class DownloadMultiOrderEmailListAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.VIEWER,)

    HEADER: List[str] = ["Номер строки", "Email", "Оформлен возврат"]

    def __init__(self, uid: int, order_id: int):
        super().__init__()
        self.uid: int = uid
        self.order_id: int = order_id

    async def _create_csv(self) -> List[bytes]:
        writer, output = create_csv_writer()
        writer.writerow(self.HEADER)
        rows = [output.getvalue().encode('utf-8')]

        row_id = 1
        async for order in self.storage.order.find(
            FindOrderParams(
                uid=self.uid,
                parent_order_id=self.order_id,
            ),
            iterator=True
        ):
            output.truncate(0)
            output.seek(0)
            refund_issued = 'Да' if order.kind == OrderKind.REFUND else '-'
            writer.writerow([row_id, order.user_email, refund_issued])
            row_id += 1
            rows.append(output.getvalue().encode('utf-8'))
        return rows

    async def handle(self) -> List[bytes]:
        return await self._create_csv()
