from copy import deepcopy
from datetime import datetime, timedelta
from decimal import Decimal
from typing import Any, Dict, Iterable, List, Optional, Tuple, Union

from sendr_utils import alist, json_value, utcnow

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.interactions.trust import GetTrustCredentialParamsAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.download_image import DownloadImageAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.actions.shop.get_or_ensure_default import GetOrEnsureDefaultShopAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import (
    PAYMETHOD_ID_OFFLINE, AcquirerType, MerchantOAuthMode, MerchantRole, OperationKind, OrderKind, OrderSource,
    PaymentsTestCase, PayStatus, ReceiptType, ShopType
)
from mail.payments.payments.core.entities.image import Image
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order, OrderData
from mail.payments.payments.core.entities.product import Product
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.entities.shop import Shop
from mail.payments.payments.core.exceptions import (
    BadCommissionError, CommissionDeniedError, OrderAbandonDeadlineError, OrderAbandonProlongationAmountError,
    OrderAlreadyHaveTransactions, OrderInvalidKind, OrderNotFoundError, OrderSourceDeniedError,
    PaymentWithout3dsNotAllowed, PaymethodIdNotFound, RecurrentPaymentModeNotAllowed
)
from mail.payments.payments.storage.exceptions import OrderNotFound


class CoreCreateOrUpdateOrderAction(BaseOrderAction, BaseMerchantAction):
    """
    Create or Update order entity.

    default_shop_type has meaning of previously existed merchant_oauth_mode
    from API point of view they are passed uniformly via ``mode=(prod|test)`` argument
    About shop_id argument:
        shop_id is set to order only when it is created for the first time.
        Subsequent updates should ignore shop_id argument and remain order.shop_id unchanged.
        When creating order:
            if shop_id is specified action verifies shop exists and sets it to order
            otherwise create default shop with type specified by default_shop_type

    If shop_id is explicitly specified, default_shop_type makes no sense.
    """
    transact = True
    skip_data = True
    skip_parent = False

    def __init__(self,
                 uid: int,
                 source: OrderSource,
                 items: Union[Iterable[dict], Iterable[Item]],
                 caption: Optional[str] = None,
                 autoclear: bool = True,
                 shop_id: Optional[int] = None,
                 default_shop_type: Optional[ShopType] = None,
                 description: Optional[str] = None,
                 user_email: Optional[str] = None,
                 customer_uid: Optional[int] = None,
                 user_description: Optional[str] = None,
                 return_url: Optional[str] = None,
                 paymethod_id: Optional[str] = None,
                 test: Optional[PaymentsTestCase] = None,
                 customer_subscription_id: Optional[int] = None,
                 kind: OrderKind = OrderKind.PAY,
                 parent_order_id: Optional[int] = None,
                 max_amount: Optional[int] = None,
                 create_order_extra: Optional[Dict] = None,
                 update_log_extra: Optional[Dict] = None,
                 order_id: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 commission: Optional[int] = None,
                 meta: Optional[str] = None,
                 offline_abandon_deadline: Optional[datetime] = None,
                 service_data: Optional[dict] = None,
                 acquirer: Optional[AcquirerType] = None,
                 receipt_type: ReceiptType = ReceiptType.COMPLETE,
                 fast_moderation: bool = False,
                 recurrent: bool = False,
                 without_3ds: bool = False,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        assert self.uid is not None

        self.shop_id = shop_id
        self.default_shop_type: ShopType = default_shop_type or ShopType.PROD
        self.caption = caption
        self.items: List[Item] = [
            (
                Item(
                    uid=self.uid,
                    amount=item['amount'],
                    product=Product(
                        uid=self.uid,
                        name=item['name'],
                        price=item['price'],
                        nds=item['nds'],
                        currency=item['currency'],
                    ),
                    image=Image(
                        uid=self.uid,
                        url=item['image']['url'],
                    ) if item.get('image') is not None else None,
                    markup=item.get('markup'),
                )
                if isinstance(item, dict) else deepcopy(item)
            )
            for item in items
        ]
        self.autoclear = autoclear
        self.description = description
        self.user_email = user_email
        self.customer_uid = customer_uid
        self.user_description = user_description
        self.return_url = return_url
        self.paymethod_id = paymethod_id
        self.test = test
        self.customer_subscription_id = customer_subscription_id
        self.kind = kind
        self.parent_order_id = parent_order_id
        self.max_amount = max_amount
        self.create_order_extra = create_order_extra or {}
        self.update_log_extra = update_log_extra or {}
        self.order_id = order_id
        self.meta = meta
        self.offline_abandon_deadline = offline_abandon_deadline
        self.service_data = service_data
        self.source = source
        self.acquirer = acquirer
        self.receipt_type = receipt_type
        self.commission = commission
        self.fast_moderation = fast_moderation
        self.recurrent = recurrent
        self.without_3ds = without_3ds

        self._shop: Shop = None  # type: ignore
        self._existing_order: Optional[Order] = None

    async def _create_order_instance(self) -> Order:
        assert self.merchant
        assert self._shop and self._shop.shop_id is not None

        paymethod_id = self.paymethod_id
        offline_abandon_deadline: Optional[datetime] = None

        if self._existing_order:
            existing_order = self._existing_order
            shop_id = existing_order.shop_id
            merchant_oauth_mode = existing_order.merchant_oauth_mode
            pay_status = existing_order.pay_status
            kind = existing_order.kind
            customer_subscription_id = existing_order.customer_subscription_id

            data = existing_order.data
            data.meta = self.meta
            data.receipt_type = self.receipt_type
            data.fast_moderation = self.fast_moderation
            data.recurrent = self.recurrent
            data.without_3ds = self.without_3ds

            offline_abandon_deadline = existing_order.offline_abandon_deadline
            if self.service_data is not None:
                data.service_data = self.service_data

            if existing_order.paymethod_id == PAYMETHOD_ID_OFFLINE:
                paymethod_id = existing_order.paymethod_id
        else:
            # Write-once fields
            shop_id = self._shop.shop_id
            merchant_oauth_mode = MerchantOAuthMode.from_shop_type(self._shop.shop_type)
            assert self.kind in [OrderKind.PAY, OrderKind.MULTI]
            if self.kind != OrderKind.MULTI:
                assert self.max_amount is None

            pay_status = PayStatus.NEW if self.kind == OrderKind.PAY else None
            kind = self.kind
            customer_subscription_id = self.customer_subscription_id

            data = OrderData(
                multi_max_amount=self.max_amount,
                meta=self.meta,
                service_data=self.service_data,
                receipt_type=self.receipt_type,
                fast_moderation=self.fast_moderation,
                recurrent=self.recurrent,
                without_3ds=self.without_3ds,
            )

        if self.kind == OrderKind.PAY and paymethod_id == PAYMETHOD_ID_OFFLINE:
            offline_abandon_deadline, data.offline_prolongation_amount = self._calculate_offline_abandon_params(
                data,
                existing_offline_abandon_deadline=offline_abandon_deadline,
            )

        commission = self.commission if self.commission is not None else self._get_payments_commission()
        order = Order(
            shop_id=shop_id,  # not updated
            paymethod_id=paymethod_id,  # not updated, if offline
            kind=kind,  # not updated
            pay_status=pay_status,  # not updated
            customer_subscription_id=customer_subscription_id,  # not updated
            data=data,  # not updated
            uid=self.merchant.uid,
            parent_order_id=self.parent_order_id,
            autoclear=self.autoclear,
            caption=self.caption,
            description=self.description,
            user_description=self.user_description,
            user_email=self.user_email,
            customer_uid=self.customer_uid,
            return_url=self.return_url,
            test=self.test,
            commission=commission,
            # пока проставляем merchant_oauth_mode для возможности отката
            # можно выкинуть поле merchant_oauth_mode после полного перехода на тип магазина
            merchant_oauth_mode=merchant_oauth_mode,
            order_id=self.order_id,
            offline_abandon_deadline=offline_abandon_deadline,
            acquirer=self.acquirer,
            created_by_source=self.source,
            **self.create_order_extra
        )
        order.shop = self._shop
        order.exclude_stats = order.is_test

        if settings.SERVICE_COMMISSION_ENABLED:
            self._check_commission(order)
        if self.acquirer is not None:
            # Проверяем наличие у мерчанта необходимых реквизитов
            await GetTrustCredentialParamsAction(acquirer=self.acquirer, merchant=self.merchant, order=order).run()

        return order

    def _get_payments_commission(self) -> int:
        return int(Decimal(settings.PAYMENTS_COMMISSION) * 100 * 100)

    def _check_commission(self, order: Order) -> None:
        if order.commission is None:
            return
        payments_commission = self._get_payments_commission()
        if self.source != OrderSource.SERVICE and order.commission != payments_commission:
            raise CommissionDeniedError

        minimal_commission = payments_commission
        if order.commission >= 100 * 100 or order.commission < minimal_commission:
            raise BadCommissionError

    def _calculate_offline_abandon_params(
        self,
        data: OrderData,
        existing_offline_abandon_deadline: Optional[datetime],
    ) -> Tuple[Optional[datetime], Optional[int]]:
        assert self.merchant

        offline_abandon_deadline: Optional[datetime] = None
        offline_prolongation_amount = data.offline_prolongation_amount

        if self.offline_abandon_deadline:
            # PAYBACK-550
            # Проверяем, что прошло времени от выставления заказу offline не больше, чем
            # в конфиге или в конфиге мерчанта. Если мы мерчанту проставили дефолтное значение больше,
            # чем в конфиге, то это значит это особенный мерчант, и ему можно превышать конфиг.
            # Если дефолт меньше (а в 99,9% будет равным), то опираемся на конфиг.
            delta = self.offline_abandon_deadline - (existing_offline_abandon_deadline or utcnow())
            max_period = max(
                settings.ORDER_OFFLINE_ABANDON_PERIOD_MAX,
                self.merchant.options.order_offline_abandon_period
            )

            if delta.total_seconds() > max_period:
                raise OrderAbandonDeadlineError(
                    params={
                        'period': delta.total_seconds(),
                        'max_period': settings.ORDER_OFFLINE_ABANDON_PERIOD_MAX
                    }
                )

            offline_abandon_deadline = self.offline_abandon_deadline

            if offline_prolongation_amount is None:
                offline_prolongation_amount = 0

            if offline_prolongation_amount >= settings.ORDER_OFFLINE_PROLONGATION_AMOUNT_MAX:
                raise OrderAbandonProlongationAmountError(
                    params={
                        'amount': offline_prolongation_amount,
                        'max_amount': settings.ORDER_OFFLINE_PROLONGATION_AMOUNT_MAX
                    }
                )
            else:
                offline_prolongation_amount += 1
        elif existing_offline_abandon_deadline is None:
            delta = timedelta(seconds=self.merchant.options.order_offline_abandon_period)
            offline_abandon_deadline = utcnow() + delta
        return offline_abandon_deadline, offline_prolongation_amount

    async def _process_order_items(self, order: Order) -> List[Item]:
        assert order.order_id is not None
        existing_items: List[Item] = await alist(
            self.storage.item.get_for_order(order.uid, order.order_id, for_update=True)
        )

        order.items = []
        for item in self.items:
            assert item.product is not None
            product, _ = await self.storage.product.get_or_create(item.product)
            item.order_id = order.order_id
            item.product_id = product.product_id
            image = item.image

            if image is not None:
                if image.image_id is None:
                    image = await self.storage.image.create(image)
                    if settings.ORDER_DOWNLOAD_IMAGES and not image.downloaded:
                        assert image.image_id is not None
                        assert self.uid is not None
                        await DownloadImageAction(uid=self.uid, image_id=image.image_id).run_async()
                item.image_id = image.image_id

            item = await self.storage.item.create_or_update(item)
            item.image = image
            item.product = product
            order.items.append(item)

        # Удаляем из базы items, которые не были переданы в текущем запросе
        actual_product_keys = [item.product.key for item in order.items]  # type: ignore
        for existing_item in existing_items:
            if existing_item.product.key not in actual_product_keys:  # type: ignore
                await self.storage.item.delete(existing_item)

        return order.items

    async def pre_handle(self) -> None:
        await super().pre_handle()
        assert (
            self.uid
            and self.merchant
        )

        if not self.merchant.options.is_creating_order_allowed(self.source):
            raise OrderSourceDeniedError

        # fetch order if exists and shop entity
        if self.order_id:
            try:
                existing_order = await self.storage.order.get(
                    self.merchant.uid,
                    self.order_id,
                    active=True,
                    select_customer_subscription=None,
                    for_update=True
                )
            except OrderNotFound:
                raise OrderNotFoundError

            assert existing_order.shop_id is not None, "Order without shop_id"
            assert existing_order.shop is not None
            self._existing_order = existing_order
            self._shop = existing_order.shop
        else:
            shop = await GetOrEnsureDefaultShopAction(
                uid=self.merchant.uid,
                shop_id=self.shop_id,
                default_shop_type=self.default_shop_type,
            ).run()
            self._shop = shop

        assert self._shop, "Existing order without Shop or no shop for new order fetched"

        trust = self.clients.get_trust_client(self.uid, self._shop.shop_type)
        if self.merchant.acquirer is not None and trust.get_merchant_setting(self.uid, 'currency_from_trust'):
            test_merchant_currency = await trust.get_enabled_currency(uid=self.uid, acquirer=self.merchant.acquirer)
            for item in self.items:
                assert item.product
                item.product.currency = test_merchant_currency

    async def handle(self) -> Order:
        assert self.merchant

        for item in self.items:
            assert item.product is not None
            item.product.adjust_nds()
        self._validate_order_items(self.items, self.merchant)

        if self._existing_order:
            existing_order = self._existing_order
            order_instance = await self._create_order_instance()

            if order_instance.service_merchant_id != existing_order.service_merchant_id:
                raise OrderNotFoundError
            if existing_order.kind != OrderKind.PAY or existing_order.customer_subscription_id is not None:
                raise OrderInvalidKind

            assert existing_order.order_id
            tx = await self.storage.transaction.get_last_by_order(
                existing_order.uid,
                existing_order.order_id,
                raise_=False
            )
            if tx:
                raise OrderAlreadyHaveTransactions

            order = await self.storage.order.save(order_instance)
        else:
            order_instance = await self._create_order_instance()
            order = await self.storage.order.create(order_instance)
        assert order.order_id is not None
        await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()

        items = await self._process_order_items(order)

        await self.pushers.log.push(
            self._create_log_instance(
                self.merchant, order,
                is_update=self.order_id is not None,
                extra=self.update_log_extra
            )
        )

        if self.order_id:
            await self.storage.change_log.create(ChangeLog(
                uid=self.merchant.uid,
                revision=order.revision,  # type: ignore
                operation=OperationKind.UPDATE_ORDER,
                arguments=json_value(self._init_kwargs),
            ))

        self.logger.context_push(order_id=order.order_id)
        self.logger.info(f'Order {"updated" if self.order_id else "created"}')

        order = await self.storage.order.get(
            order.uid,
            order.order_id,
            select_customer_subscription=bool(self.customer_subscription_id)
        )
        order.items = items
        order.add_crypto(settings.CRYPTO_V1_F1_PREFIX, self.crypto)

        return order


class CreateOrUpdateOrderAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)
    transact = True

    def __init__(self, uid: int, **kwargs: Any):
        super().__init__()
        self.uid = uid
        self.kwargs = kwargs

    async def handle(self) -> Order:
        return await CoreCreateOrUpdateOrderAction(uid=self.uid, source=OrderSource.UI, **self.kwargs).run()


class CreateOrUpdateOrderServiceMerchantAction(AuthServiceMerchantMixin, BaseDBAction):
    transact = True

    def __init__(self,
                 service_merchant_id: Optional[int] = None,
                 service_tvm_id: Optional[int] = None,
                 service: Optional[Service] = None,
                 uid: Optional[int] = None,
                 recurrent: bool = False,
                 without_3ds: bool = False,
                 **kwargs: Any,
                 ):
        self.service_merchant_id: Optional[int] = service_merchant_id
        self.service_tvm_id: Optional[int] = service_tvm_id
        self.service: Optional[Service] = service
        self.uid = uid
        self.recurrent = recurrent
        self.without_3ds = without_3ds
        self.kwargs = kwargs
        super().__init__()

    async def handle(self) -> Order:
        assert self.uid is not None
        assert self.service is not None

        if self.recurrent and not self.service.options.allow_payment_mode_recurrent:
            raise RecurrentPaymentModeNotAllowed

        if self.without_3ds and not self.service.options.allow_payment_without_3ds:
            raise PaymentWithout3dsNotAllowed

        if (self.recurrent or self.without_3ds) and 'paymethod_id' not in self.kwargs:
            raise PaymethodIdNotFound

        create_order_extra = {
            'service_client_id': self.service_client_id,
            'service_merchant_id': self.service_merchant_id
        }

        update_log_extra = {
            'service_id': self.service.service_id if self.service else None,
            'service_name': self.service.name if self.service else None
        }

        return await CoreCreateOrUpdateOrderAction(uid=self.uid,
                                                   source=OrderSource.SERVICE,
                                                   create_order_extra=create_order_extra,
                                                   update_log_extra=update_log_extra,
                                                   recurrent=self.recurrent,
                                                   without_3ds=self.without_3ds,
                                                   **self.kwargs).run()
