import hashlib
from copy import deepcopy
from typing import Tuple

from sendr_utils import alist

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.entities.image import Image
from mail.payments.payments.storage.exceptions import ImageHashAlreadyExistsStorageError


class DownloadImageAction(BaseDBAction):
    action_name = 'download_image'
    transact = True

    def __init__(self, uid: int, image_id: int) -> None:
        super().__init__()
        self.uid = uid
        self.image_id = image_id

    @staticmethod
    def _get_digest(data: bytes) -> Tuple[str, str]:
        return hashlib.md5(data).hexdigest(), hashlib.sha256(data).hexdigest()

    async def _update_dependants(self, image: Image, image_orig: Image) -> None:
        items = await alist(
            self.storage.item.find_by_image(uid=self.uid, image_id=image_orig.image_id, for_update=True)
        )
        orders = {}
        for item in items:
            key = (item.uid, item.order_id)
            if key not in orders:
                orders[key] = await self.storage.order.get(uid=item.uid, order_id=item.order_id, for_update=True)

            item.image_id = image.image_id
            await self.storage.item.save(item)

        for order in orders.values():
            assert order.order_id is not None
            await self.storage.order.save(order)  # Сохраняем order, чтобы апнуть его ревизию
            await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()

        if image.image_id != image_orig.image_id:
            await self.storage.image.delete(image_orig)

    async def handle(self) -> None:
        image = await self.storage.image.get(uid=self.uid, image_id=self.image_id, for_update=True)
        if image.stored_path is not None:
            return

        image_bytes = await self.clients.zora_images.get_image(image.url)
        md5, sha256 = self._get_digest(image_bytes)

        image_orig: Image = deepcopy(image)
        try:
            image.md5 = md5
            image.sha256 = sha256
            image = await self.storage.image.save(image)
        except ImageHashAlreadyExistsStorageError:
            image = await self.storage.image.get_by_digest(uid=self.uid, md5=md5, sha256=sha256, for_update=True)

        if image.stored_path is None:
            image.stored_path = (await self.clients.avatars.upload(image_bytes)).url

        await self.storage.image.save(image)  # Сохраняем, чтобы увеличился updated

        await self._update_dependants(image=image, image_orig=image_orig)
