from typing import Any, Dict, Optional

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.interactions.so import SenderLetterIsSpamAction
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.get_trust_env import GetOrderTrustEnvAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import (
    FunctionalityType, MerchantRole, OperationKind, OrderKind, TrustEnv
)
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.exceptions import OrderAlreadyEmailedError, OrderNotFoundError, SpamError
from mail.payments.payments.storage.exceptions import OrderNotFound
from mail.payments.payments.utils.helpers import decimal_format


class OrderEmailAction(BaseMerchantAction, BaseOrderAction):
    transact = True
    skip_parent = False
    skip_data = False
    required_merchant_roles = (MerchantRole.OPERATOR,)

    def __init__(self,
                 uid: int,
                 order_id: int,
                 to_email: str,
                 reply_email: str,
                 user_ip: str,
                 spam_check: bool = False,
                 merchant: Optional[Merchant] = None,
                 select_customer_subscription: Optional[bool] = False,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.order_id: int = order_id
        self.to_email: str = to_email
        self.reply_email: str = reply_email
        self.user_ip: str = user_ip
        self.spam_check: bool = spam_check
        self.select_customer_subscription: Optional[bool] = select_customer_subscription

    async def handle(self) -> dict:
        assert self.merchant

        try:
            order = await self.storage.order.get(
                uid=self.merchant.uid,
                order_id=self.order_id,
                select_customer_subscription=self.select_customer_subscription)
        except OrderNotFound:
            raise OrderNotFoundError

        trust_env = await GetOrderTrustEnvAction(order=order).run()

        if trust_env != TrustEnv.SANDBOX:
            await self.require_moderation(self.merchant, functionality_type=FunctionalityType.PAYMENTS)

        if order.email_context is not None:
            raise OrderAlreadyEmailedError

        order.add_crypto(settings.CRYPTO_V1_F1_PREFIX, self.crypto)
        items = await self._fetch_items(order)
        sorted_items = sorted(
            filter(lambda x: x.product, items),
            key=lambda x: x.product_id  # type: ignore
        )

        render_context: Dict[str, Any] = {
            'company_name': self.merchant.organization.full_name,
            'company_short_name': self.merchant.organization.name,
            'uid': self.merchant.uid,
            'order_id': self.order_id,
            'items': [
                {
                    'number': number,
                    'name': item.product.name,  # type: ignore
                    'price': decimal_format(item.price),  # type: ignore
                    'amount': decimal_format(item.amount),
                    'total': decimal_format(item.total_price),  # type: ignore
                } for number, item in enumerate(sorted_items, start=1)
            ],
            'total_price': decimal_format(self._items_price(items)),
            'pay_href': order.payments_url
        }

        email_context = {
            'mailing_id': settings.SENDER_MAILING_ORDER_EMAIL,
            'to_email': self.to_email,
            'render_context': render_context,
            'reply_email': self.reply_email,
        }

        if self.spam_check:
            form_is_spam = await SenderLetterIsSpamAction(
                mailing_id=settings.SENDER_MAILING_ORDER_EMAIL,
                to_email=self.to_email,
                render_context=render_context,
                user_ip=self.user_ip,
                from_email=self.reply_email,
                from_uid=self.merchant.uid
            ).run()

            if form_is_spam:
                raise SpamError

        self.logger.context_push(order_id=order.order_id, uid=order.uid)
        self.logger.info('Send email')

        email_message_id = await self.clients.sender.send_transactional_letter(**email_context)

        if order.kind == OrderKind.PAY:
            order.email_message_id = email_message_id
            order.email_context = email_context

        order = await self.storage.order.save(order)
        assert order.order_id is not None
        await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()
        assert order.revision

        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=order.revision,
            operation=OperationKind.SEND_ORDER_EMAIL,
            arguments={'email_message_id': email_message_id},
            info=email_context,
        ))

        return email_context
