import asyncio
from typing import Dict, List, Optional, Union

from sendr_utils import alist

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.get_timeline import GetOrderTimelineAction
from mail.payments.payments.core.entities.enums import MerchantRole, OrderKind, TransactionStatus
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.exceptions import OrderNotFoundError
from mail.payments.payments.storage.exceptions import OrderNotFound
from mail.payments.payments.storage.mappers.order.order import FindOrderParams


class BaseGetOrderAction(BaseOrderAction):
    def __init__(self,
                 skip_add_crypto: bool = False,
                 with_refunds: bool = False,
                 with_timeline: bool = False,
                 for_update: bool = False,
                 ):
        super().__init__()
        self.skip_add_crypto = skip_add_crypto
        self.with_refunds = with_refunds
        self.with_timeline = with_timeline
        self.order_for_update = for_update

    def _get_kwargs(self) -> dict:
        raise NotImplementedError

    def _get_order_not_found_exception(self) -> OrderNotFoundError:
        raise NotImplementedError

    async def handle(self) -> Order:
        try:
            kwargs: dict = self._get_kwargs()
            kwargs['for_update'] = self.order_for_update
            order = await self.storage.order.get(**kwargs)
        except OrderNotFound:
            raise self._get_order_not_found_exception()

        assert order.order_id

        transaction = await self.storage.transaction.get_last_by_order(
            uid=order.uid,
            order_id=order.order_id,
            raise_=False,
        )

        if transaction is not None:
            order.trust_resp_code = transaction.trust_resp_code
            order.trust_payment_id = transaction.trust_payment_id
            if transaction.status == TransactionStatus.ACTIVE:
                order.trust_url = transaction.trust_payment_url

        if not self.skip_add_crypto:
            order.add_crypto(settings.CRYPTO_V1_F1_PREFIX, self.crypto)

        if self.with_refunds:
            order.refunds = await alist(self.storage.order.find(
                FindOrderParams(uid=order.uid, original_order_id=order.order_id)
            ))

        order.items = await self._fetch_items(order)

        if self.with_timeline:
            order.timeline = await GetOrderTimelineAction(order).run()
        return order


class CoreGetOrderAction(BaseGetOrderAction):
    def __init__(self,
                 uid: int,
                 order_id: int,
                 original_order_id: Optional[int] = None,
                 customer_uid: Optional[int] = None,
                 skip_add_crypto: bool = False,
                 for_update: bool = False,
                 kind: Optional[OrderKind] = None,
                 with_customer_subscription: bool = False,
                 with_refunds: bool = False,
                 with_timeline: bool = False,
                 select_customer_subscription: Optional[bool] = False,
                 ):
        super().__init__(skip_add_crypto=skip_add_crypto, with_refunds=with_refunds, with_timeline=with_timeline,
                         for_update=for_update)
        self.uid = uid
        self.order_id = order_id
        self.original_order_id = original_order_id
        self.customer_uid = customer_uid
        self.kind = kind
        self.with_customer_subscription = with_customer_subscription
        self.select_customer_subscription = select_customer_subscription

    def _get_kwargs(self) -> dict:
        kwargs: Dict[str, Union[Optional[int], bool, OrderKind]] = {
            'uid': self.uid,
            'order_id': self.order_id,
            'original_order_id': self.original_order_id,
            'customer_uid': self.customer_uid,
            'with_customer_subscription': self.with_customer_subscription,
            'select_customer_subscription': self.select_customer_subscription
        }
        if self.kind is not None:
            kwargs['kind'] = self.kind
        return kwargs

    def _get_order_not_found_exception(self) -> OrderNotFoundError:
        return OrderNotFoundError(uid=self.uid, order_id=self.order_id)


class GetOrderAction(CoreGetOrderAction):
    required_merchant_roles = (MerchantRole.VIEWER,)


class GetOrderByCustomerSubscriptionIdAction(BaseGetOrderAction):
    def __init__(self,
                 uid: int,
                 customer_subscription_id: int,
                 customer_uid: Optional[int] = None,
                 skip_add_crypto: bool = False,
                 with_refunds: bool = False,
                 for_update: bool = False,
                 ):
        super().__init__(skip_add_crypto=skip_add_crypto, with_refunds=with_refunds, for_update=for_update)
        self.uid = uid
        self.customer_subscription_id = customer_subscription_id
        self.customer_uid = customer_uid

    def _get_kwargs(self) -> dict:
        return {
            'uid': self.uid,
            'with_customer_subscription': True,
            'customer_subscription_id': self.customer_subscription_id,
            'customer_uid': self.customer_uid,
        }

    def _get_order_not_found_exception(self) -> OrderNotFoundError:
        raise OrderNotFoundError(
            uid=self.uid,
            customer_subscription_id=self.customer_subscription_id,
        )


class GetOrderServiceMerchantAction(AuthServiceMerchantMixin, BaseGetOrderAction):
    def __init__(self,
                 service_tvm_id: int,
                 service_merchant_id: int,
                 order_id: int,
                 original_order_id: Optional[int] = None,
                 customer_uid: Optional[int] = None,
                 skip_add_crypto: bool = False,
                 for_update: bool = False,
                 kind: Optional[OrderKind] = None,
                 with_customer_subscription: bool = False,
                 with_refunds: bool = False,
                 select_customer_subscription: bool = False,
                 ):
        super().__init__(skip_add_crypto=skip_add_crypto, with_refunds=with_refunds, for_update=for_update)

        self.service_tvm_id = service_tvm_id
        self.service_merchant_id = service_merchant_id
        self.order_id = order_id
        self.original_order_id = original_order_id
        self.customer_uid = customer_uid
        self.kind = kind
        self.with_customer_subscription = with_customer_subscription
        self.with_refunds = with_refunds
        self.select_customer_subscription = select_customer_subscription

    def _get_kwargs(self) -> dict:
        kwargs: Dict[str, Union[Optional[int], bool, OrderKind]] = {
            'uid': self.uid,
            'order_id': self.order_id,
            'original_order_id': self.original_order_id,
            'customer_uid': self.customer_uid,
            'service_merchant_id': self.service_merchant_id,
            'with_customer_subscription': self.with_customer_subscription,
            'select_customer_subscription': self.select_customer_subscription
        }
        if self.kind is not None:
            kwargs['kind'] = self.kind
        return kwargs

    def _get_order_not_found_exception(self) -> OrderNotFoundError:
        return OrderNotFoundError(
            uid=self.uid,
            order_id=self.order_id,
            service_merchant_id=self.service_merchant_id,
        )


class GetInternalOrderPayoutInfoAction(AuthServiceMerchantMixin, BaseOrderAction):
    def __init__(self,
                 service_tvm_id: int,
                 service_merchant_id: int,
                 order_id: int,
                 ):
        super().__init__()

        self.service_tvm_id = service_tvm_id
        self.service_merchant_id = service_merchant_id
        self.order_id = order_id

    async def handle(self) -> List[dict]:
        assert self.uid
        order = await self.storage.order.get(
            uid=self.uid,
            order_id=self.order_id,
            service_merchant_id=self.service_merchant_id
        )
        assert order.order_id
        transaction = await self.storage.transaction.get_last_by_order(
            uid=order.uid,
            order_id=order.order_id,
            raise_=False,
        )
        if transaction and transaction.status == TransactionStatus.CLEARED:
            data = await self.clients.balance_http.get_payouts_by_purchase_token(transaction.trust_purchase_token)
            payouts = data.get('payouts', [])
            if payouts:
                return payouts

            payments = data.get('payments', [])
            child_purchase_tokens = []
            for payment in payments:
                child_purchase_tokens += payment.get('child_payments', [])

            data = await asyncio.gather(*[
                self.clients.balance_http.get_payouts_by_purchase_token(child_purchase_token)
                for child_purchase_token in child_purchase_tokens
            ])
            for item in data:
                payouts += item.get('payouts', [])

            return payouts

        return []
