from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.update_transaction import UpdateTransactionAction
from mail.payments.payments.core.exceptions import CoreNotFoundError, OrderNotFoundError
from mail.payments.payments.http_helpers.crypto import DecryptionError, InvalidToken
from mail.payments.payments.storage.exceptions import OrderNotFound
from mail.payments.payments.utils.helpers import masked_exception


class GetOrderByHashAction(BaseOrderAction, BaseMerchantAction):
    manual_load = True
    transact = True

    def __init__(self, hash_: str):
        super().__init__()
        self.hash_: str = hash_

    async def handle(self) -> dict:
        with masked_exception(
                OrderNotFoundError,
                logger=self.logger,
                ignore_exceptions=(DecryptionError, InvalidToken, CoreNotFoundError),
        ):
            with self.crypto.decrypt_order(self.hash_) as data:
                uid = data['uid']
                order_id = data['order_id']

            try:
                await self.load_merchant(uid=uid)
                order = await self.storage.order.get(uid=uid,
                                                     order_id=order_id,
                                                     for_update=True,
                                                     with_customer_subscription=True,
                                                     select_customer_subscription=None,)
                order.items = await self._fetch_items(order)
                transaction = await self.storage.transaction.get_last_by_order(
                    uid=uid,
                    order_id=order_id,
                    raise_=False,
                    for_update=True,
                )
            except OrderNotFound:
                raise OrderNotFoundError(uid=uid, order_id=order_id)

            # Run update
            if transaction is not None and not transaction.finished:
                transaction = await UpdateTransactionAction(transaction=transaction).run()

            if transaction is not None:
                order.trust_resp_code = transaction.trust_resp_code
                order.trust_payment_id = transaction.trust_payment_id

        order.add_crypto(settings.CRYPTO_V1_F1_PREFIX, self.crypto)
        return {
            'order': order,
            'merchant': self.merchant,
            'transaction': transaction,
        }
