from collections import OrderedDict
from datetime import datetime
from decimal import Decimal
from typing import Any, Iterable, List, MutableSequence, Optional, Union

from sendr_utils import alist

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.entities.enums import MerchantRole, OrderKind, OrderSource, PayStatus, RefundStatus
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.storage.mappers.order.order import FindOrderParams


class OrderListResult(list, MutableSequence[Order]):
    """List of Orders returned by List action with extra optional info"""
    # total count of found orders without limit restrictions
    total_found_count: Optional[int] = None


class CoreGetOrderListAction(BaseOrderAction):
    def __init__(self,
                 uid: Optional[int] = None,
                 order_id: Optional[int] = None,
                 original_order_id: Optional[int] = None,
                 created_from: Optional[datetime] = None,
                 created_to: Optional[datetime] = None,
                 updated_from: Optional[datetime] = None,
                 updated_to: Optional[datetime] = None,
                 held_at_from: Optional[datetime] = None,
                 held_at_to: Optional[datetime] = None,
                 kinds: Optional[Iterable[OrderKind]] = None,
                 pay_method: Optional[str] = None,
                 pay_statuses: Optional[Iterable[PayStatus]] = None,
                 refund_statuses: Optional[Iterable[RefundStatus]] = None,
                 price_from: Optional[Decimal] = None,
                 price_to: Optional[Decimal] = None,
                 is_active: Optional[bool] = None,
                 text_query: Optional[str] = None,
                 parent_order_id: Optional[int] = None,
                 email_query: Optional[str] = None,
                 sort_by: Optional[str] = None,
                 descending: Optional[bool] = None,
                 limit: Optional[int] = None,
                 offset: Optional[int] = None,
                 created_by_sources: Optional[Iterable[OrderSource]] = None,
                 service_ids: Optional[Iterable[int]] = None,
                 subscription: Optional[bool] = False,
                 count_found: bool = False,
                 with_refunds: bool = False,
                 ):
        super().__init__()
        self.uid: Optional[int] = uid
        self.order_id: Optional[int] = order_id
        self.original_order_id: Optional[int] = original_order_id
        self.created_from: Optional[datetime] = created_from
        self.created_to: Optional[datetime] = created_to
        self.updated_from: Optional[datetime] = updated_from
        self.updated_to: Optional[datetime] = updated_to
        self.held_at_from: Optional[datetime] = held_at_from
        self.held_at_to: Optional[datetime] = held_at_to
        self.kinds: Optional[Iterable[OrderKind]] = kinds
        self.pay_statuses: Optional[Iterable[PayStatus]] = pay_statuses
        self.refund_statuses: Optional[Iterable[RefundStatus]] = refund_statuses
        self.price_from: Optional[Decimal] = price_from
        self.price_to: Optional[Decimal] = price_to
        self.is_active: Optional[bool] = is_active
        self.text_query: Optional[str] = text_query
        self.parent_order_id: Optional[int] = parent_order_id
        self.email_query: Optional[str] = email_query
        self.sort_by: Optional[str] = sort_by
        self.descending: Optional[bool] = descending
        self.limit: Optional[int] = limit
        self.offset: Optional[int] = offset
        self.pay_method: Optional[str] = pay_method
        self.created_by_sources: Optional[Iterable[OrderSource]] = created_by_sources
        self.service_ids: Optional[Iterable[int]] = service_ids
        self.subscription: Optional[bool] = subscription
        self.count_found = count_found
        self.with_refunds = with_refunds

    def _get_order_key_value(self, uid: int, order_id: Optional[int]) -> str:
        return f'{uid}-{order_id}'

    def _get_order_key(self, value: Union[Order, Item]) -> str:
        return self._get_order_key_value(value.uid, value.order_id)

    def _get_found_count_params(self) -> FindOrderParams:
        return FindOrderParams(
            uid=self.uid,
            order_id=self.order_id,
            original_order_id=self.original_order_id,
            created_from=self.created_from,
            created_to=self.created_to,
            updated_from=self.updated_from,
            updated_to=self.updated_to,
            held_at_from=self.held_at_from,
            held_at_to=self.held_at_to,
            price_from=self.price_from,
            price_to=self.price_to,
            kinds=self.kinds,
            pay_statuses=self.pay_statuses,
            refund_statuses=self.refund_statuses,
            is_active=self.is_active,
            text_query=self.text_query,
            parent_order_id=self.parent_order_id,
            email_query=self.email_query,
            pay_method=self.pay_method,
            created_by_sources=self.created_by_sources,
            service_ids=self.service_ids,
            with_customer_subscription=self.subscription is not False,
            select_customer_subscription=self.subscription,
        )

    def _get_find_order_params(self) -> FindOrderParams:
        find_params = self._get_found_count_params()
        find_params.limit = self.limit
        find_params.offset = self.offset
        find_params.sort_by = self.sort_by
        find_params.descending = self.descending
        return find_params

    async def handle(self) -> OrderListResult:
        # Getting orders, keeping their order
        find_params = self._get_find_order_params()
        orders_by_key = OrderedDict(
            [
                (self._get_order_key(order), order)
                async for order in
                self.storage.order.find(find_params)
            ]
        )

        # set refunds to empty list instead of None if necessary
        for order in orders_by_key.values():
            order.refunds = order.refunds or list()

        uid_and_order_id_list = [
            (order.uid, order.order_id) for order in list(orders_by_key.values()) if order.order_id
        ]
        async for item in self.storage.item.get_for_orders(uid_and_order_id_list):
            key = self._get_order_key(item)
            if orders_by_key[key].items is None:
                orders_by_key[key].items = []
            orders_by_key[key].items.append(item)  # type: ignore

        if self.with_refunds:
            refunds_by_key = OrderedDict(
                [
                    (self._get_order_key(refund), refund)
                    async for refund in self.storage.order.get_refunds_for_orders(uid_and_order_id_list)
                ]
            )

            refunds_uid_and_and_order_id_list = [
                (refund.uid, refund.order_id) for refund in list(refunds_by_key.values()) if refund.order_id
            ]
            async for refund_item in self.storage.item.get_for_orders(refunds_uid_and_and_order_id_list):
                key = self._get_order_key(refund_item)
                if refunds_by_key[key].items is None:
                    refunds_by_key[key].items = []
                refunds_by_key[key].items.append(refund_item)  # type: ignore

            for refund in refunds_by_key.values():
                key = self._get_order_key_value(refund.uid, refund.original_order_id)
                orders_by_key[key].refunds.append(refund)  # type: ignore

        orders: List[Order] = list(orders_by_key.values())
        transactions = await alist(self.storage.transaction.find_last_by_orders(uid_and_order_id_list))
        uid_tx_mapping = {tx.order_id: tx for tx in transactions}

        for order in orders:
            assert order.order_id
            order.add_crypto(settings.CRYPTO_V1_F1_PREFIX, self.crypto)

            transaction = uid_tx_mapping.get(order.order_id)
            if transaction is not None:
                order.trust_resp_code = transaction.trust_resp_code
                order.trust_payment_id = transaction.trust_payment_id

        result = OrderListResult(orders)

        if self.count_found:
            found_count_params = self._get_found_count_params()
            result.total_found_count = await self.storage.order.get_found_count(found_count_params)

        return result


class GetOrderListAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.VIEWER,)

    def __init__(self, **kwargs):
        super().__init__()
        self.kwargs = kwargs

    async def handle(self) -> OrderListResult:
        return await CoreGetOrderListAction(**self.kwargs).run()


class GetServiceMerchantOrderListAction(AuthServiceMerchantMixin, CoreGetOrderListAction):
    def __init__(self, service_tvm_id: int, service_merchant_id: int, *args: Any, **kwargs: Any):
        super().__init__(*args, **kwargs)
        self.service_tvm_id: int = service_tvm_id
        self.service_merchant_id: int = service_merchant_id

    def _get_find_order_params(self) -> FindOrderParams:
        params = super()._get_find_order_params()
        params.service_merchant_id = self.service_merchant_id
        return params
