from decimal import Decimal
from typing import Any, Awaitable, Callable, Dict, Iterable, Optional, Tuple, TypedDict, Union, cast

from sendr_qlog import LoggerContext
from sendr_utils import enum_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.interactions.trust import (
    GetTrustCredentialParamsAction, ProcessNewOrderInTrustAction
)
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.get_trust_env import GetOrderTrustEnvAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.actions.update_transaction import UpdateTransactionAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.enums import (
    PAYMETHOD_ID_OFFLINE, AcquirerType, FunctionalityType, OrderKind, OrderSource, PayStatus, TransactionStatus,
    TrustEnv
)
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.log import OrderPaymentStartedLog
from mail.payments.payments.core.entities.merchant import Merchant, PaymentSystemsOptions
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.entities.transaction import Transaction
from mail.payments.payments.core.exceptions import (
    CoreFailError, CoreNotFoundError, MerchantNotFoundError, OrderAbandonedError, OrderAlreadyPaidError,
    OrderArchivedError, OrderCancelledError, OrderCannotBePaidWithoutEmailError, OrderCannotBePaidWithoutReturnUrlError,
    OrderNotAllowedByModerationPolicyError, OrderNotFoundError, OrderRefundCannotBePaidError,
    ServiceMerchantNotEnabledError, ServiceMerchantNotFoundError
)
from mail.payments.payments.http_helpers.crypto import Crypto, DecryptionError, InvalidToken
from mail.payments.payments.interactions.trust.entities import PaymentMode
from mail.payments.payments.storage.exceptions import MerchantNotFound, OrderNotFound, ServiceMerchantNotFound
from mail.payments.payments.utils.helpers import masked_exception


def decrypt_payment(crypto: Crypto, hash_: str, logger: LoggerContext = None) -> Tuple[int, int]:
    with masked_exception(
        OrderNotFoundError,
        logger=logger,
        ignore_exceptions=(DecryptionError, InvalidToken, CoreNotFoundError),
    ):
        with crypto.decrypt_payment(hash_) as data:
            return data['uid'], data['order_id']


class CorePayOrderActionResult(TypedDict):
    payment_url: Optional[str]
    purchase_token: Optional[str]
    acquirer: Optional[AcquirerType]
    payment_systems_options: PaymentSystemsOptions
    environment: TrustEnv
    merchant: Merchant


class CorePayOrderAction(APICallbackTaskMixin, BaseOrderAction, BaseMerchantAction):
    transact = True
    manual_load = True
    skip_moderation = True

    def __init__(self,
                 order_id: Optional[int] = None,
                 hash_: Optional[str] = None,
                 template: Optional[str] = None,
                 trust_form_name: Optional[str] = None,
                 return_url: Optional[str] = None,
                 customer_uid: Optional[int] = None,
                 yandexuid: Optional[str] = None,
                 login_id: Optional[str] = None,
                 email: Optional[str] = None,
                 description: Optional[str] = None,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 referer: Optional[str] = None,
                 user_agent: Optional[str] = None,
                 payment_mode: Optional[PaymentMode] = PaymentMode.WEB_PAYMENT,
                 paymethod_id: Optional[str] = 'trust_web_page',
                 pay_by_source: OrderSource = OrderSource.UI,
                 service_data: Optional[dict] = None,
                 order_service_merchant_id: Optional[int] = None,
                 turboapp_id: Optional[str] = None,
                 tsid: Optional[str] = None,
                 psuid: Optional[str] = None,
                 order_by_hash: bool = True,
                 trust_params_getter: Optional[Callable[[Order], Awaitable[dict]]] = None,
                 order_getter: Optional[Callable[[], Awaitable[Order]]] = None,
                 overwrite_return_url: bool = True,
                 payment_completion_action: Optional[Union[str, Dict[str, str]]] = None,
                 select_customer_subscription: Optional[bool] = False,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.yandexuid = yandexuid
        self.login_id = login_id
        self.order_id = order_id
        self.hash_ = hash_
        self.template = template
        self.trust_form_name = trust_form_name
        self.return_url = return_url
        self.customer_uid = customer_uid
        self.email = email
        self.description = description
        self.referer = referer
        self.user_agent = user_agent
        self.payment_mode = payment_mode
        self.paymethod_id = paymethod_id
        self.pay_by_source = pay_by_source
        self.service_data = service_data
        self.order_service_merchant_id = order_service_merchant_id
        self.trust_params_getter = trust_params_getter
        self.order_getter = order_getter
        self.order_by_hash = order_by_hash
        self.overwrite_return_url = overwrite_return_url
        self.payment_completion_action = payment_completion_action
        self.select_customer_subscription = select_customer_subscription

        # only for logging
        self.turboapp_id = turboapp_id
        self.tsid = tsid
        self.psuid = psuid

    async def _get_order(self) -> Order:
        if self.order_getter is not None:
            order = await self.order_getter()
        else:
            assert self.merchant is not None
            try:
                order = await self.storage.order.get(
                    uid=self.merchant.uid,
                    order_id=self.order_id,
                    for_update=True,
                    select_customer_subscription=self.select_customer_subscription
                )
            except OrderNotFound:
                raise OrderNotFoundError(uid=self.merchant.uid, order_id=self.order_id)

        if order.shop is None:
            assert order.shop_id
            order.shop = await self.storage.shop.get(uid=order.uid, shop_id=order.shop_id)

        return order

    async def _get_service(self, order: Order) -> Optional[Service]:
        service: Optional[Service] = None

        if order.service_merchant_id is not None:
            if order.service_client_id:
                service = await self.storage.service.get_by_related(service_client_id=order.service_client_id,
                                                                    service_merchant_id=order.service_merchant_id)
            else:
                service_merchant = await self.storage.service_merchant.get(order.service_merchant_id)
                service = cast(Service, service_merchant.service)
        return service

    @staticmethod
    def _check_order(order: Order) -> None:
        if order.kind != OrderKind.PAY:
            raise OrderRefundCannotBePaidError(params={'kind': enum_value(order.kind)})
        if order.is_already_paid:
            raise OrderAlreadyPaidError
        if order.pay_status == PayStatus.MODERATION_NEGATIVE:
            raise OrderNotAllowedByModerationPolicyError
        if order.pay_status == PayStatus.ABANDONED:
            raise OrderAbandonedError
        if order.pay_status == PayStatus.CANCELLED:
            raise OrderCancelledError
        if not order.active:
            raise OrderArchivedError

    def _check_trust_params(self, order: Order) -> None:
        if (
            self.payment_mode in [PaymentMode.WEB_PAYMENT, PaymentMode.EXTERNAL_WEB_PAYMENT]
            and self.return_url is None
            and order.return_url is None
        ):
            raise OrderCannotBePaidWithoutReturnUrlError

        if self.email is None and order.user_email is None:
            raise OrderCannotBePaidWithoutEmailError

    async def _initialize_transaction(self, order: Order, items: Iterable[Item],
                                      service: Optional[Service]) -> Transaction:
        template_tag = 'desktop/form'
        if self.template is None or self.template == 'mobile':
            template_tag = 'mobile/form'

        return_url = order.return_url
        if not return_url or (self.overwrite_return_url and self.return_url):
            return_url = self.return_url
        assert self.merchant is not None

        if order.is_subscription:
            assert order.customer_subscription and order.customer_subscription.subscription
            acquirer = order.customer_subscription.acquirer
        else:
            acquirer = self.merchant.acquirer

        acquirer, _, _ = await GetTrustCredentialParamsAction(
            acquirer=acquirer,
            merchant=self.merchant,
            order=order
        ).run()

        order.acquirer = acquirer

        trust_params: dict = {
            'merchant': self.merchant,
            'order': order,
            'items': items,
            'return_url': return_url,
            'trust_form_name': self.trust_form_name,
            'template_tag': template_tag,
            'yandexuid': self.yandexuid,
            'login_id': self.login_id,
            'customer_uid': self.customer_uid or order.customer_uid,
            'user_email': self.email or order.user_email,
            'paymethod_id': order.paymethod_id or self.paymethod_id,
            'payment_mode': self.payment_mode,
            'payment_completion_action': self.payment_completion_action,
        }

        if service is not None:
            if (service_fee := service.options.service_fee) is not None:
                trust_params['service_fee'] = service_fee
            if settings.SERVICE_COMMISSION_ENABLED:
                if (commission := service.options.commission) is not None:
                    trust_params['commission'] = commission
                else:
                    trust_params['commission'] = str(Decimal(settings.PAYMENTS_COMMISSION) * 100 * 100)
            if service.antifraud:
                # TODO: проставить всем slug и выпилить этот assert
                assert service.slug, 'Slug should be set when antifraud is enabled'
                trust_params['payments_service_slug'] = service.slug

        if settings.SERVICE_COMMISSION_ENABLED and order.commission is not None:
            trust_params['commission'] = order.commission

        if self.trust_params_getter is not None:
            trust_params.update(await self.trust_params_getter(order))

        trust_response = await ProcessNewOrderInTrustAction(**trust_params).run()

        await self.storage.order.save(order)

        self.logger.info('Transaction initialized in trust')

        assert order.order_id is not None
        assert self.merchant is not None

        return Transaction(
            uid=self.merchant.uid,
            order_id=order.order_id,
            trust_purchase_token=trust_response['purchase_token'],
            trust_payment_url=trust_response.get('payment_url'),
            trust_payment_id=trust_response['trust_payment_id'],
        )

    async def _log_status_update(self, order: Order, transaction: Transaction) -> None:
        assert (
            order.order_id is not None
            and self.merchant is not None
            and transaction.trust_purchase_token is not None
        )
        service = None
        if order.service_client_id is not None and order.service_merchant_id is not None:
            service = await self.storage.service.get_by_related(
                service_client_id=order.service_client_id,
                service_merchant_id=order.service_merchant_id,
            )
        service_id = service.service_id if service else None
        service_name = service.name if service else None

        order.items = await self._fetch_items(order)

        log = OrderPaymentStartedLog(
            merchant_name=self.merchant.name,
            merchant_uid=self.merchant.uid,
            merchant_acquirer=order.get_acquirer(self.merchant.acquirer),
            order_id=order.order_id,
            price=order.log_price,
            purchase_token=transaction.trust_purchase_token,
            referer=self.referer,
            user_agent=self.user_agent,
            customer_uid=order.customer_uid,
            service_id=service_id,
            service_name=service_name,
            sdk_api_created=order.created_by_source == OrderSource.SDK_API,
            sdk_api_pay=order.pay_by_source == OrderSource.SDK_API,
            created_by_source=order.created_by_source,
            pay_by_source=order.pay_by_source,
            merchant_oauth_mode=enum_value(order.merchant_oauth_mode),
            turboapp_id=self.turboapp_id,
            tsid=self.tsid,
            psuid=self.psuid,
        )
        if self.hash_:
            log.pay_token = settings.PAY_TOKEN_PREFIX + self.hash_

        await self.pushers.log.push(log)

    async def _create_new_transaction(self, order: Order, items: Iterable[Item],
                                      service: Optional[Service]) -> Transaction:
        # Updating order
        pay_status = order.pay_status

        order.pay_status = PayStatus.NEW
        order.user_email = self.email if self.email else order.user_email
        order.user_description = self.description if self.description else order.user_description
        order.data.trust_form_name = self.trust_form_name
        order.data.trust_template = self.template
        order.data.turboapp_id = self.turboapp_id
        order.data.tsid = self.tsid
        order.data.psuid = self.psuid
        order.pay_by_source = self.pay_by_source

        if self.service_data is not None:
            order.data.service_data = self.service_data
        if self.order_service_merchant_id is not None:
            order.service_merchant_id = self.order_service_merchant_id

        if self.customer_uid is not None:
            order.customer_uid = self.customer_uid
        if order.paymethod_id == PAYMETHOD_ID_OFFLINE:
            order.paymethod_id = None

        order = await self.storage.order.save(order)
        order = await self.storage.order.get(
            order.uid,
            order.order_id,
            select_customer_subscription=None,
            with_customer_subscription=True
        )
        assert order.order_id is not None
        await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()

        assert order.revision is not None
        assert order.pay_status is not None

        await self.storage.change_log.create(ChangeLog(
            uid=order.uid,
            revision=order.revision,
            operation=OperationKind.UPDATE_ORDER,
            arguments={
                'order_id': order.order_id,
                'pay_status': enum_value(order.pay_status),
                'user_email': order.user_email,
            },
        ))

        with self.logger:
            self.logger.context_push(revision=order.revision)
            self.logger.info('Order update: pay_status')

        # Scheduling callback if order had rejected status
        if pay_status == PayStatus.REJECTED:
            if service is not None and service.service_client:
                await self.create_order_status_task(order=order, service=service)
            assert self.merchant is not None
            await self.create_order_status_task(order=order, merchant=self.merchant)
            self.logger.info('Order status changed from REJECTED to NEW. Callback is scheduled')

        transaction = await self._initialize_transaction(order, items, service)
        transaction = await self.storage.transaction.create(transaction)
        with self.logger:
            self.logger.context_push(revision=transaction.revision, tx_id=transaction.tx_id)
            self.logger.info('Transaction created')

        await self._log_status_update(order, transaction)

        return transaction

    async def handle(self) -> CorePayOrderActionResult:
        self.logger.context_push(yandexuid=self.yandexuid)

        if self.order_by_hash and not self.order_id:
            assert self.hash_
            self.uid, self.order_id = decrypt_payment(self.crypto, self.hash_, self.logger)
            self.logger.context_push(uid=self.uid, order_id=self.order_id)

        try:
            await self.load_merchant(uid=self.uid, merchant=self.merchant)
            order = await self._get_order()
            service = await self._get_service(order)
            items = await self._fetch_items(order)
            assert self.merchant and order.order_id
            transaction = await self.storage.transaction.get_last_by_order(
                uid=self.merchant.uid,
                order_id=order.order_id,
                raise_=False,
                for_update=True,
            )
        except MerchantNotFound:
            raise MerchantNotFoundError(uid=self.uid)

        assert self.merchant

        trust_env = await GetOrderTrustEnvAction(order=order).run()
        if trust_env != TrustEnv.SANDBOX:
            await self.require_moderation(self.merchant, functionality_type=FunctionalityType.PAYMENTS)

        # Run update
        if transaction is not None and not transaction.finished:
            self.logger.context_push(tx_id=transaction.tx_id)
            self.logger.info('Not finished transaction found. Updating')
            transaction = await UpdateTransactionAction(transaction=transaction).run()

        # Check order
        self._check_order(order)
        self._check_trust_params(order)

        if transaction is None or transaction.status in (TransactionStatus.CANCELLED, TransactionStatus.FAILED):
            transaction = await self._create_new_transaction(order, items, service)
        elif transaction.status == TransactionStatus.ACTIVE:
            self.logger.info('Transaction is active and can be paid')
        else:
            self.logger.context_push(order_pay_status=order.pay_status, transaction_status=transaction.status)
            self.logger.error('Order transaction status mismatch')
            raise CoreFailError('Order transaction status mismatch')

        return {
            'payment_url': transaction.trust_payment_url,
            'purchase_token': transaction.trust_purchase_token,
            'acquirer': order.get_acquirer(self.merchant.acquirer),
            'payment_systems_options': self.merchant.options.payment_systems,
            'environment': trust_env,
            'merchant': self.merchant,
        }


class PayOrderByHashAction(BaseDBAction):
    transact = True

    def __init__(self, **kwargs: Any):
        super().__init__()
        self.kwargs = kwargs

    async def handle(self) -> CorePayOrderActionResult:
        return await CorePayOrderAction(**self.kwargs).run()


class StartOrderAction(AuthServiceMerchantMixin, BaseDBAction):
    transact = True

    def __init__(self, service_tvm_id: int, service_merchant_id: int, **kwargs: Any):
        super().__init__()
        self.service_tvm_id = service_tvm_id
        self.service_merchant_id = service_merchant_id
        self.kwargs = kwargs

    async def _trust_params(self, order: Order) -> dict:
        result: Dict[str, Union[Optional[str], int, PaymentMode]] = {}

        if order.paymethod_id:
            result.update({
                'paymethod_id': 'trust_web_page',
                'selected_card_id': order.paymethod_id
            })
        if order.data.without_3ds:
            result.update({
                'payment_mode': PaymentMode.API_PAYMENT,
                'paymethod_id': order.paymethod_id
            })
        if order.data.recurrent:
            result.update({
                'payment_mode': PaymentMode.RECURRING,
                'paymethod_id': order.paymethod_id
            })

        return result

    async def handle(self) -> CorePayOrderActionResult:
        return await CorePayOrderAction(uid=self.uid, trust_params_getter=self._trust_params, **self.kwargs).run()


class StartOrderByPayTokenAction(BaseDBAction):
    transact = True

    def __init__(self,
                 pay_token: str,
                 email: Optional[str] = None,
                 customer_uid: Optional[int] = None,
                 pay_by_source: OrderSource = OrderSource.UI,
                 turboapp_id: Optional[str] = None,
                 tsid: Optional[str] = None,
                 psuid: Optional[str] = None,
                 ):
        super().__init__()
        self.hash_ = pay_token[len(settings.PAY_TOKEN_PREFIX):]
        self.email = email
        self.customer_uid = customer_uid
        self.pay_by_source = pay_by_source
        self.turboapp_id = turboapp_id
        self.tsid = tsid
        self.psuid = psuid

    async def handle(self) -> CorePayOrderActionResult:
        service_merchant_id: Optional[int] = None

        # PAYBACK-556
        if settings.TURBOAPP_SERVICE_ID is None:
            self.logger.warning('TURBOAPP_SERVICE_ID is not set.')
        elif self.turboapp_id is None:
            self.logger.info('Turboapp_id is not passed.')
        else:
            uid, order_id = decrypt_payment(self.crypto, self.hash_, self.logger)
            try:
                service_merchant = await self.storage.service_merchant.get(
                    uid=uid,
                    service_id=settings.TURBOAPP_SERVICE_ID,
                    entity_id=self.turboapp_id,
                )
            except ServiceMerchantNotFound:
                raise ServiceMerchantNotFoundError
            if not service_merchant.enabled:
                raise ServiceMerchantNotEnabledError
            service_merchant_id = service_merchant.service_merchant_id
            self.logger.context_push(service_merchant_id=service_merchant_id)
            self.logger.info('Found service_merchant for order by turboapp_id.')
        if self.email is None:
            self.email = ""

        return await CorePayOrderAction(
            hash_=self.hash_,
            email=self.email,
            paymethod_id=None,
            payment_mode=None,
            customer_uid=self.customer_uid,
            pay_by_source=self.pay_by_source,
            order_service_merchant_id=service_merchant_id,
            turboapp_id=self.turboapp_id,
            tsid=self.tsid,
            psuid=self.psuid,
        ).run()
