from typing import Optional

from sendr_utils import enum_value, json_value, utcnow

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import (
    PAYMETHOD_ID_OFFLINE, MerchantRole, OperationKind, OrderKind, OrderSource, PayStatus, TransactionStatus
)
from mail.payments.payments.core.entities.log import OrderStatusUpdatedLog
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.exceptions import (
    MerchantNotFoundError, OrderAbandonedError, OrderAlreadyHaveTransactions, OrderAlreadyPaidError, OrderArchivedError,
    OrderNotFoundError, OrderRefundCannotBePaidError
)
from mail.payments.payments.storage.exceptions import MerchantNotFound, OrderNotFound


class CorePayOfflineOrderAction(BaseOrderAction, BaseMerchantAction):
    transact = True
    manual_load = True
    skip_parent = False
    skip_data = False
    skip_moderation = True

    def __init__(self,
                 order_id: int,
                 customer_uid: Optional[int] = None,
                 description: Optional[str] = None,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 referer: Optional[str] = None,
                 user_agent: Optional[str] = None,
                 service_data: Optional[dict] = None,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.order_id = order_id
        self.customer_uid = customer_uid
        self.description = description
        self.referer = referer
        self.user_agent = user_agent
        self.service_data = service_data

    async def _get_order(self) -> Order:
        try:
            assert self.uid is not None
            return await self.storage.order.get(uid=self.uid, order_id=self.order_id, for_update=True)
        except OrderNotFound:
            raise OrderNotFoundError(uid=self.uid, order_id=self.order_id)

    async def _check_order(self, order: Order) -> None:
        assert order.order_id
        transaction = await self.storage.transaction.get_last_by_order(
            uid=order.uid,
            order_id=order.order_id,
            raise_=False,
        )

        if transaction and (not transaction.finished or transaction.status == TransactionStatus.CLEARED):
            raise OrderAlreadyHaveTransactions
        if order.kind != OrderKind.PAY:
            raise OrderRefundCannotBePaidError(params={'kind': enum_value(order.kind)})
        if order.is_already_paid:
            raise OrderAlreadyPaidError
        if order.pay_status == PayStatus.ABANDONED:
            raise OrderAbandonedError
        if not order.active:
            raise OrderArchivedError

    async def _log_status_update(self, order: Order) -> None:
        assert order.order_id is not None
        assert self.merchant is not None

        service: Optional[Service] = None
        if order.service_client_id is not None and order.service_merchant_id is not None:
            service = await self.storage.service.get_by_related(
                service_client_id=order.service_client_id,
                service_merchant_id=order.service_merchant_id,
            )

        service_id = service.service_id if service else None
        service_name = service.name if service else None

        await self.pushers.log.push(
            OrderStatusUpdatedLog(
                merchant_name=self.merchant.name,
                merchant_uid=self.merchant.uid,
                merchant_acquirer=order.get_acquirer(self.merchant.acquirer),
                order_id=order.order_id,
                purchase_token=None,
                status=enum_value(order.pay_status),
                sdk_api_created=order.created_by_source == OrderSource.SDK_API,
                sdk_api_pay=order.pay_by_source == OrderSource.SDK_API,
                created_by_source=order.created_by_source,
                pay_by_source=order.pay_by_source,
                customer_uid=order.customer_uid,
                service_id=service_id,
                service_name=service_name,
                referer=self.referer,
                user_agent=self.user_agent,
                merchant_oauth_mode=enum_value(order.merchant_oauth_mode),
                price=order.log_price
            )
        )

    async def handle(self) -> Order:
        try:
            await self.load_merchant(self.uid)
        except MerchantNotFound:
            raise MerchantNotFoundError(uid=self.uid)
        assert self.merchant

        order: Order = await self._get_order()
        await self._check_order(order)

        order.pay_status = PayStatus.PAID
        order.closed = utcnow()
        order.paymethod_id = PAYMETHOD_ID_OFFLINE
        order.user_description = self.description if self.description else order.user_description
        if self.customer_uid is not None:
            order.customer_uid = self.customer_uid
        if self.service_data is not None:
            order.data.service_data = self.service_data

        order = await self.storage.order.save(order)
        assert order.order_id is not None
        await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()
        await self._log_status_update(order)
        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=order.revision,  # type: ignore
            operation=OperationKind.UPDATE_ORDER,
            arguments=json_value({
                'pay_status': order.pay_status,
                'paymethod_id': order.paymethod_id,
                **self._init_kwargs
            }),
        ))

        return order


class PayOfflineOrderAction(CorePayOfflineOrderAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)


class PayOfflineServiceMerchantOrderAction(AuthServiceMerchantMixin, CorePayOfflineOrderAction):
    def __init__(self,
                 service_tvm_id: int,
                 service_merchant_id: int,
                 order_id: int,
                 customer_uid: Optional[int] = None,
                 description: Optional[str] = None,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 service_data: Optional[dict] = None,
                 ):
        super().__init__(
            order_id=order_id,
            customer_uid=customer_uid,
            description=description,
            uid=uid,
            merchant=merchant,
            service_data=service_data,
        )
        self.service_tvm_id: int = service_tvm_id
        self.service_merchant_id: int = service_merchant_id
