from typing import Any, Dict, List, Optional, cast

from sendr_utils import json_value

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get_acquirer import GetAcquirerMerchantAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.get import CoreGetOrderAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import (
    PAY_METHOD_YANDEX, AcquirerType, OperationKind, OrderKind, PayStatus, ReceiptType
)
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.exceptions import (
    ItemsInvalidDataError, OrderInvalidPayMethod, OrderMustBePaidError, OrderReceiptTypeMustPrepaid,
    TransactionNotFoundError, TrustError
)
from mail.payments.payments.interactions.trust.exceptions import TrustException
from mail.payments.payments.storage.exceptions import TransactionNotFound


class CoreOrderReceiptCloseAction(BaseDBAction):  # PAYBACK-589
    def __init__(self, uid: int, order_id: int, items: Optional[List[Dict[str, Any]]]):
        super().__init__()
        self.uid = uid
        self.order_id = order_id
        self.items = items

    async def handle(self) -> dict:
        order: Order = await CoreGetOrderAction(uid=self.uid, order_id=self.order_id, kind=OrderKind.PAY).run()
        assert order.shop is not None
        acquirer: AcquirerType = await GetAcquirerMerchantAction(uid=self.uid).run()

        try:
            tx = await self.storage.transaction.get_last_by_order(self.uid, self.order_id)
        except TransactionNotFound:
            raise TransactionNotFoundError

        assert tx and tx.trust_purchase_token

        self.logger.context_push(uid=self.uid, order_id=self.order_id, purchase_token=tx.trust_purchase_token)

        if order.pay_status != PayStatus.PAID:
            raise OrderMustBePaidError
        elif order.data.receipt_type != ReceiptType.PREPAID:
            raise OrderReceiptTypeMustPrepaid
        elif order.pay_method != PAY_METHOD_YANDEX:
            raise OrderInvalidPayMethod(params={'pay_method': order.pay_method})

        items_from_db = {
            item.product_id: item
            async for item in self.storage.item.get_for_order(self.uid, self.order_id)
        }
        items: List[Item] = []

        if self.items is not None:
            for item in self.items:
                product_id = item['product_id']
                if product_id not in items_from_db:
                    raise ItemsInvalidDataError(params={'product_id': product_id})

                item_from_db = items_from_db[product_id]
                assert item_from_db.product
                if item.get('nds'):
                    item_from_db.product.nds = item['nds']

                items.append(item_from_db)
        else:
            items += items_from_db.values()

        trust = self.clients.get_trust_client(self.uid, order.shop.shop_type)

        try:
            result = await trust.payment_deliver(
                uid=self.uid,
                acquirer=acquirer,
                purchase_token=tx.trust_purchase_token,
                order=order,
                items=items
            )
        except TrustException as e:
            with self.logger:
                self.logger.context_push(status_code=e.message)
                self.logger.warning('Trust receipt close error.')
            raise TrustError(params={'status_code': e.message})
        else:
            await self.storage.change_log.create(ChangeLog(
                uid=order.uid,
                revision=cast(int, order.revision),
                operation=OperationKind.RECEIPT_CLOSE,
                arguments={
                    'order_id': order.order_id,
                    'items': json_value(self.items),
                }
            ))
            self.logger.info('Receipt close.')

        return result


class OrderReceiptCloseServiceMerchantAction(AuthServiceMerchantMixin, BaseDBAction):
    def __init__(self,
                 service_merchant_id: Optional[int] = None,
                 service_tvm_id: Optional[int] = None,
                 service: Optional[Service] = None,
                 **kwargs: Any,
                 ):
        self.service_merchant_id: Optional[int] = service_merchant_id
        self.service_tvm_id: Optional[int] = service_tvm_id
        self.service: Optional[Service] = service
        self.kwargs = kwargs
        super().__init__()

    async def handle(self) -> dict:
        assert self.uid
        return await CoreOrderReceiptCloseAction(uid=self.uid, **self.kwargs).run()
