# coding: utf-8
from decimal import Decimal
from typing import Dict, Iterable, List, Optional, Tuple

from sendr_utils import alist, enum_value

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.interactions.trust import (
    GetPaymentInfoInTrustAction, GetTrustCredentialParamsAction
)
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.order.base import BaseOrderAction
from mail.payments.payments.core.actions.order.get_trust_env import GetOrderTrustEnvAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.customer_subscription import CustomerSubscription
from mail.payments.payments.core.entities.customer_subscription_transaction import CustomerSubscriptionTransaction
from mail.payments.payments.core.entities.enums import (
    PAYMETHOD_ID_OFFLINE, FunctionalityType, MerchantRole, OrderKind, OrderSource, PaymentsTestCase, PayStatus,
    RefundStatus, TaskType, TransactionStatus, TrustEnv
)
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.log import RefundCreatedLog
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.not_fetched import NotFetchedType
from mail.payments.payments.core.entities.order import Order, OrderData
from mail.payments.payments.core.entities.product import Product, ProductKeyType
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.entities.task import Task
from mail.payments.payments.core.entities.transaction import Transaction
from mail.payments.payments.core.exceptions import (
    CoreActionDenyError, CustomerSubscriptionNotFoundError, CustomerSubscriptionTransactionHasOpenRefundsError,
    CustomerSubscriptionTransactionNotClearedError, CustomerSubscriptionTransactionNotFoundError,
    OrderItemNotPresentInOriginalOrderError, OrderNotFoundError, OrderOriginalOrderMustBePaidError,
    OrderOriginalOrderMustBePayKind, OrderRequestedItemAmountExceedsPaidError, ServiceNotFoundError,
    TestOrderCannotBeRefundedError, TransactionNotFoundError
)
from mail.payments.payments.storage.exceptions import (
    CustomerSubscriptionNotFound, CustomerSubscriptionTransactionNotFound, OrderNotFound, ServiceNotFound,
    TransactionNotFound
)
from mail.payments.payments.storage.mappers.order.order import FindOrderParams


class BaseCoreCreateRefundAction(APICallbackTaskMixin, BaseOrderAction, BaseMerchantAction):
    """
    Creates refund in db and trust. Starts it in an asynchronous task
    """
    transact = True
    skip_data = True
    skip_parent = False

    def __init__(self,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 caption: Optional[str] = None,
                 description: Optional[str] = None,
                 meta: Optional[str] = None):
        super().__init__(uid=uid, merchant=merchant)
        self.caption = caption
        self.description: Optional[str] = description
        self.meta: Optional[str] = meta

    def _create_refund_instance(self, original_order: Order) -> Order:
        assert self.merchant
        return Order(
            uid=self.merchant.uid,
            original_order_id=original_order.order_id,
            shop_id=original_order.shop_id,
            caption=self.caption or f'Возврат по заказу №{original_order.order_id}',
            description=self.description,
            kind=OrderKind.REFUND,
            pay_status=None,
            data=OrderData(meta=self.meta),
            refund_status=RefundStatus.CREATED,
            acquirer=original_order.acquirer,
            service_merchant_id=original_order.service_merchant_id,
            service_client_id=original_order.service_client_id,
        )

    async def _create_db_instances(self, original_order: Order,
                                   src_items: Iterable[Item]) -> Tuple[Order, List[Item]]:
        # create refund in database
        refund = await self.storage.order.create(self._create_refund_instance(original_order))
        refund.shop = await self.storage.shop.get(uid=refund.uid, shop_id=refund.shop_id)

        # creating refund items in database
        items = []
        for item in src_items:
            item.order_id = refund.order_id
            created_item = await self.storage.item.create(item)
            created_item.product = item.product
            items.append(created_item)

        return refund, items

    async def _get_service(self, refund: Order) -> Service:
        try:
            return await self.storage.service.get_by_related(service_client_id=refund.service_client_id,
                                                             service_merchant_id=refund.service_merchant_id)
        except ServiceNotFound:
            raise ServiceNotFoundError(service_client_id=refund.service_client_id,
                                       service_merchant_id=refund.service_merchant_id)

    async def _schedule_callbacks(self, refund: Order) -> None:
        await self.create_refund_status_task(refund, merchant=self.merchant)
        if refund.service_merchant_id is not None and refund.service_client_id is not None:
            service = await self._get_service(refund)
            await self.create_refund_status_task(refund, service=service)

    async def _save_and_send_to_history(self, refund: Order) -> Order:
        refund = await self.storage.order.save(refund)
        assert refund.original_order_id is not None
        await SendToHistoryOrderAction(uid=refund.uid, order_id=refund.original_order_id).run_async()
        return refund

    async def _start_refund_task(self, refund: Order) -> None:
        if refund.trust_refund_id:
            # Create START_REFUND task
            assert refund.order_id is not None
            task = await self.storage.task.create(
                Task(
                    task_type=TaskType.START_REFUND,
                    params=dict(uid=refund.uid, order_id=refund.order_id)
                )
            )
            self.logger.context_push(task_id=task.task_id)
        await self._schedule_callbacks(refund)

    async def _log_refund(self, refund: Order, order: Order) -> None:
        # Pushing
        assert (
            refund.original_order_id is not None
            and refund.order_id is not None
            and refund.refund_status is not None
        )

        # Logging
        assert self.merchant
        assert refund.revision is not None
        assert refund.items
        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=refund.revision,
            operation=OperationKind.ADD_REFUND,
            arguments={'order_id': order.order_id, 'refund_id': refund.order_id}
        ))

        await self.pushers.log.push(
            RefundCreatedLog(
                merchant_name=self.merchant.name,
                merchant_uid=self.merchant.uid,
                merchant_acquirer=order.get_acquirer(self.merchant.acquirer),
                refund_id=refund.order_id,
                order_id=refund.original_order_id,
                status=enum_value(refund.refund_status),
                price=refund.log_price,
                items=[item.dump() for item in refund.items],
                paymethod_id=order.paymethod_id,
            )
        )

        self.logger.info('Refund created')


class CoreCreateRefundAction(BaseCoreCreateRefundAction):
    def __init__(self,
                 order_id: int,
                 items: Iterable[dict],
                 caption: Optional[str] = None,
                 description: Optional[str] = None,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 meta: Optional[str] = None
                 ):
        super().__init__(uid=uid, merchant=merchant, caption=caption, description=description, meta=meta)
        self.order_id: int = order_id
        self.item_dicts: Iterable[dict] = items

    async def _get_order(self) -> Order:
        assert self.merchant
        order = await self.storage.order.get(uid=self.merchant.uid, order_id=self.order_id, for_update=True)
        return order

    async def _refund_validation(self, order: Order, refund_items: Iterable[Item]) -> None:
        assert order.order_id is not None and order.items is not None and self.merchant
        assert all(item.product is not None for item in order.items)

        items: Dict[ProductKeyType, Item] = {
            item.product.key: item
            for item in order.items
            if item.product is not None
        }
        already_refunded_products: Dict[int, Decimal] = {
            product_id: amount
            async for product_id, amount in self.storage.item.get_product_amount_in_refunds(
                uid=order.uid,
                order_id=order.order_id
            )
        }

        for refund_item in refund_items:
            assert refund_item.product is not None
            key = refund_item.product.key

            # check refunded product matches original order products
            if key not in items:
                raise OrderItemNotPresentInOriginalOrderError
            item_by_key = items[key]
            assert item_by_key is not None and item_by_key.product is not None

            refund_item.product_id = item_by_key.product.product_id
            refund_item.product = item_by_key.product

            # check refunded product amount doesn't exceed amount in original order
            assert refund_item.product_id
            available_for_refund = item_by_key.amount - already_refunded_products.get(refund_item.product_id, 0)
            if refund_item.amount > available_for_refund:
                raise OrderRequestedItemAmountExceedsPaidError

        if order.pay_method != PAYMETHOD_ID_OFFLINE and order.get_acquirer(self.merchant.acquirer) is None:
            raise CoreActionDenyError

        trust_env = await GetOrderTrustEnvAction(order=order).run()
        if trust_env != TrustEnv.SANDBOX:
            await self.require_moderation(self.merchant, functionality_type=FunctionalityType.PAYMENTS)

    async def _fetch_entities(self) -> Tuple[Order, Optional[Transaction]]:
        assert self.merchant is not None

        try:
            order = await self._get_order()
            order.items = await self._fetch_items(order)
            self.logger.context_push(paymethod_id=order.paymethod_id)
        except OrderNotFound:
            self.logger.info('Refund is not created: order not found')
            raise OrderNotFoundError(uid=self.merchant.uid, order_id=self.order_id)

        tx: Optional[Transaction] = None
        if order.paymethod_id != PAYMETHOD_ID_OFFLINE:
            try:
                tx = await self.storage.transaction.get_last_by_order(uid=self.merchant.uid, order_id=self.order_id)
                assert tx is not None
                self.logger.context_push(tx_id=tx.tx_id)
            except TransactionNotFound:
                self.logger.info('Refund is not created: order has no transactions')
                raise TransactionNotFoundError(uid=self.merchant.uid, order_id=self.order_id)

        return order, tx

    def _create_items(self) -> Iterable[Item]:
        assert self.uid
        items: List[Item] = [
            Item(
                uid=self.uid,
                order_id=None,
                product_id=None,
                amount=item_data['amount'],
                product=Product(
                    uid=self.uid,
                    name=item_data['name'],
                    price=item_data['price'],
                    nds=item_data['nds'],
                    currency=item_data['currency'],
                )
            )
            for item_data in self.item_dicts
        ]
        for item in items:
            assert item.product is not None
            item.product.adjust_nds()
        return items

    async def handle(self) -> Order:
        assert self.uid is not None and self.merchant is not None
        self.logger.context_push(order_id=self.order_id, uid=self.uid)

        # Generate items for refund
        self.items = self._create_items()
        self._validate_order_items(self.items, self.merchant)

        order, tx = await self._fetch_entities()

        if order.test is not None:
            if order.test == PaymentsTestCase.TEST_OK_CLEAR:
                tx = None  # process as if paymethod_id == 'offline'
            else:
                self.logger.info('Test order can not be refunded')
                raise TestOrderCannotBeRefundedError

        if order.kind != OrderKind.PAY:
            self.logger.info(f'Refund is not created: order is not {OrderKind.PAY.value} kind')
            raise OrderOriginalOrderMustBePayKind
        elif order.pay_status != PayStatus.PAID:
            self.logger.info('Refund is not created: order is not paid')
            raise OrderOriginalOrderMustBePaidError

        await self._refund_validation(order, self.items)

        # Creating refund and it's items in database
        refund, items = await self._create_db_instances(order, self.items)

        if tx is not None:
            assert (
                tx.trust_purchase_token is not None
                and order.order_id is not None
                and order.shop is not None
                and not isinstance(self.merchant.oauth, NotFetchedType)
                and self.merchant.acquirer is not None
            )

            acquirer = order.get_acquirer(self.merchant.acquirer)
            assert acquirer is not None

            payment_info = await GetPaymentInfoInTrustAction(
                purchase_token=tx.trust_purchase_token,
                order=order,
                acquirer=order.get_acquirer(self.merchant.acquirer),  # type: ignore
            ).run()

            trust = self.clients.get_trust_client(order.uid, order.shop.shop_type)

            acquirer_type, submerchant_id, oauth = await GetTrustCredentialParamsAction(
                acquirer=acquirer,
                merchant=self.merchant,
                order=refund
            ).run()

            assert refund.caption
            refund.trust_refund_id = await trust.refund_create(
                uid=self.merchant.uid,
                original_order_id=order.order_id,
                customer_uid=payment_info.get('uid'),
                caption=refund.caption,
                purchase_token=tx.trust_purchase_token,
                items=items,
                acquirer=acquirer_type,
                submerchant_id=submerchant_id,
                oauth=oauth,
                version=order.data.version,
            )

            assert refund.trust_refund_id

            self.logger.context_push(trust_refund_id=refund.trust_refund_id)
            self.logger.info('Refund created in trust')
        else:
            refund.refund_status = RefundStatus.COMPLETED
            self.logger.info('Skip refund creation in trust')

        refund = await self._save_and_send_to_history(refund)
        refund.items = items

        await self._start_refund_task(refund)
        await self._log_refund(refund, order)

        return refund


class CreateRefundAction(CoreCreateRefundAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)


class CreateServiceMerchantRefundAction(AuthServiceMerchantMixin, CoreCreateRefundAction):
    def __init__(self,
                 service_merchant_id: int,
                 service_tvm_id: int,
                 order_id: int,
                 items: Iterable[dict],
                 caption: Optional[str] = None,
                 description: Optional[str] = None,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 ):
        self.service_merchant_id: int = service_merchant_id
        self.service_tvm_id: int = service_tvm_id
        super().__init__(
            order_id=order_id,
            items=items,
            caption=caption,
            description=description,
            uid=uid,
            merchant=merchant,
        )

    def _create_refund_instance(self, order: Order) -> Order:
        order = super()._create_refund_instance(order)
        order.service_client_id = self.service_client_id
        order.service_merchant_id = self.service_merchant_id
        order.created_by_source = OrderSource.SERVICE
        return order

    async def _get_order(self) -> Order:
        assert self.merchant
        return await self.storage.order.get(uid=self.merchant.uid,
                                            order_id=self.order_id,
                                            service_merchant_id=self.service_merchant_id)

    async def _get_service(self, refund: Order) -> Service:
        assert self.service
        return self.service

    async def handle(self) -> Order:
        self.logger.context_push(
            service_client_id=self.service_client_id,
            service_merchant_id=self.service_merchant_id,
        )
        return await super().handle()


class CreateCustomerSubscriptionTransactionRefundAction(BaseCoreCreateRefundAction):
    def __init__(self,
                 uid: int,
                 customer_subscription_id: int,
                 purchase_token: str,
                 merchant: Optional[Merchant] = None,
                 caption: Optional[str] = None,
                 description: Optional[str] = None,
                 meta: Optional[str] = None):
        super().__init__(uid=uid, merchant=merchant, caption=caption, description=description, meta=meta)
        self.customer_subscription_id = customer_subscription_id
        self.purchase_token = purchase_token
        self.customer_subscription: Optional[CustomerSubscription] = None
        self.customer_subscription_transaction: Optional[CustomerSubscriptionTransaction] = None

    async def _fetch_existing_refunds(self, original_order: Order) -> List[Order]:
        params = FindOrderParams(
            select_customer_subscription=True,
            uid=self.uid,
            original_order_id=original_order.order_id,
            customer_subscription_id=self.customer_subscription_id,
            customer_subscription_tx_purchase_token=self.purchase_token
        )
        return await alist(self.storage.order.find(params))

    async def _get_order(self, subs: CustomerSubscription) -> Order:
        assert self.merchant
        assert subs.order_id is not None

        order = await self.storage.order.get(
            uid=self.merchant.uid, order_id=subs.order_id, select_customer_subscription=True
        )
        return order

    def _create_refund_instance(self, original_order: Order) -> Order:
        assert self.customer_subscription
        assert self.customer_subscription_transaction

        refund = super()._create_refund_instance(original_order)
        refund.customer_subscription_id = self.customer_subscription.customer_subscription_id
        refund.customer_subscription_tx_purchase_token = self.customer_subscription_transaction.purchase_token

        return refund

    async def _fetch_entities(self, subs: CustomerSubscription) -> Order:
        assert self.merchant is not None

        try:
            order = await self._get_order(subs)
            order.items = await self._fetch_items(order)
            self.logger.context_push(paymethod_id=order.paymethod_id)
        except OrderNotFound:
            self.logger.info('Refund is not created: order not found')
            raise OrderNotFoundError(uid=self.merchant.uid, order_id=subs.order_id)

        return order

    def _original_order_validation(self, order: Order) -> None:
        if order.test is not None:
            self.logger.info('Test order can not be refunded')
            raise TestOrderCannotBeRefundedError

        if order.kind != OrderKind.PAY:
            self.logger.info(f'Refund is not created: order is not {OrderKind.PAY.value} kind')
            raise OrderOriginalOrderMustBePayKind
        elif order.pay_status != PayStatus.PAID:
            self.logger.info('Refund is not created: order is not paid')
            raise OrderOriginalOrderMustBePaidError

    def _refund_validation(self,
                           tx: CustomerSubscriptionTransaction,
                           existing_refunds: Iterable[Order]) -> None:
        # transaction is cleared
        if tx.payment_status != TransactionStatus.CLEARED:
            raise CustomerSubscriptionTransactionNotClearedError

        # no open or success refunds
        open_refunds = [refund for refund in existing_refunds if refund.refund_status != RefundStatus.FAILED]
        if len(open_refunds) > 0:
            raise CustomerSubscriptionTransactionHasOpenRefundsError

    async def _create_trust_refund(self, refund: Order, subs: CustomerSubscription, tx: CustomerSubscriptionTransaction,
                                   order: Order) -> Order:
        assert (
            tx.purchase_token is not None
            and order.order_id is not None
            and order.shop is not None
            and self.merchant is not None
            and not isinstance(self.merchant.oauth, NotFetchedType)
            and self.merchant.acquirer is not None
        )

        acquirer = order.get_acquirer(self.merchant.acquirer)
        assert acquirer is not None

        payment_info = await GetPaymentInfoInTrustAction(
            purchase_token=tx.purchase_token,
            order=order,
            acquirer=order.get_acquirer(self.merchant.acquirer),  # type: ignore
        ).run()

        trust = self.clients.get_trust_client(order.uid, order.shop.shop_type)

        acquirer_type, submerchant_id, oauth = await GetTrustCredentialParamsAction(
            acquirer=acquirer,
            merchant=self.merchant,
            order=refund
        ).run()

        assert refund.caption
        assert tx.trust_order_id
        refund.trust_refund_id = await trust.refund_create_single(
            uid=self.merchant.uid,
            customer_uid=payment_info.get('uid'),
            caption=refund.caption,
            purchase_token=tx.purchase_token,
            quantity=subs.quantity,
            trust_order_id=tx.trust_order_id,
            acquirer=acquirer_type,
            submerchant_id=submerchant_id,
            oauth=oauth,
        )
        assert refund.trust_refund_id

        self.logger.context_push(trust_refund_id=refund.trust_refund_id)
        self.logger.info('Refund created in trust')
        return refund

    async def handle(self) -> Order:
        assert self.uid is not None and self.customer_subscription_id is not None and self.purchase_token is not None
        assert self.merchant is not None
        self.logger.context_push(
            uid=self.uid,
            customer_subscription_id=self.customer_subscription_id,
            purchase_token=self.purchase_token
        )

        # 1. fetch customer subscription transaction
        try:
            self.customer_subscription_transaction = await self.storage.customer_subscription_transaction.get(
                uid=self.uid,
                customer_subscription_id=self.customer_subscription_id,
                purchase_token=self.purchase_token
            )
        except CustomerSubscriptionTransactionNotFound:
            self.logger.info('Refund is not created: customer subscription transaction not found')
            raise CustomerSubscriptionTransactionNotFoundError(
                uid=self.merchant.uid,
                customer_subscription_id=self.customer_subscription_id,
                purchase_token=self.purchase_token
            )

        # 2. fetch customer subscription
        try:
            self.customer_subscription = await self.storage.customer_subscription.get(
                uid=self.uid,
                customer_subscription_id=self.customer_subscription_id
            )
        except CustomerSubscriptionNotFound:
            self.logger.info('Refund is not created: customer subscription not found')
            raise CustomerSubscriptionNotFoundError(
                uid=self.merchant.uid,
                customer_subscription_id=self.customer_subscription_id,
            )

        assert self.customer_subscription
        assert self.customer_subscription_transaction

        # 3. fetch order with items
        original_order = await self._fetch_entities(self.customer_subscription)

        # 4. validate
        self._original_order_validation(original_order)
        existing_refunds = await self._fetch_existing_refunds(original_order)
        self._refund_validation(self.customer_subscription_transaction, existing_refunds)

        # 5. make refund for original order, refund items copied from order
        assert original_order.items
        refund, items = await self._create_db_instances(original_order, original_order.items)
        self.logger.context_push(refund_id=refund.order_id)

        # 6. make refund in trust
        refund = await self._create_trust_refund(
            refund, self.customer_subscription, self.customer_subscription_transaction, original_order
        )

        # 7. post TRUST interaction actions
        refund = await self._save_and_send_to_history(refund)
        refund.items = items

        await self._start_refund_task(refund)
        await self._log_refund(refund, original_order)

        return refund
