from typing import List

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get_acquirer import GetAcquirerMerchantAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import OperationKind
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.transaction import Transaction


class ResizeOrderAction(BaseDBAction):
    transact = True

    def __init__(self,
                 transaction: Transaction,
                 order: Order,
                 items: List[Item]
                 ):
        super().__init__()
        self.transaction = transaction
        self.order = order
        self.items = items

    async def handle(self) -> None:
        assert self.order.shop is not None
        acquirer = await GetAcquirerMerchantAction(uid=self.order.uid).run()
        acquirer = self.order.get_acquirer(acquirer)

        assert self.transaction.trust_purchase_token

        for item in self.items:
            trust_client = self.clients.get_trust_client(self.order.uid, self.order.shop.shop_type)
            await trust_client.payment_resize(
                uid=self.order.uid,
                acquirer=acquirer,
                purchase_token=self.transaction.trust_purchase_token,
                order=self.order,
                item=item
            )
            item = await self.storage.item.save(item)

        # save in order to update revision
        order = await self.storage.order.save(self.order)
        assert order.revision
        await self.storage.change_log.create(ChangeLog(
            uid=self.order.uid,
            revision=order.revision,
            operation=OperationKind.UPDATE_ORDER,
            arguments={
                'order_id': self.order.order_id,
                'items': [ext_item.dump() for ext_item in self.items]
            }
        ))
