from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.customer_subscription.sync import SyncCustomerSubscriptionAction
from mail.payments.payments.core.actions.update_transaction import UpdateTransactionAction
from mail.payments.payments.storage.exceptions import TransactionNotFound


class SyncOrderAction(BaseDBAction):
    transact = True

    def __init__(self, uid: int, order_id: int):
        super().__init__()
        self.uid: int = uid
        self.order_id: int = order_id

    async def handle(self) -> None:
        try:
            tx = await self.storage.transaction.get_last_by_order(
                uid=self.uid,
                order_id=self.order_id,
                for_update=True,
            )
        except TransactionNotFound:
            self.logger.debug('Order has no transactions. Nothing to sync.')
            return

        assert tx is not None

        order = await self.storage.order.get(
            uid=self.uid,
            order_id=self.order_id,
            select_customer_subscription=None
        )

        self.logger.context_push(tx_id=tx.tx_id, order_id=self.order_id, uid=self.uid)

        if not tx.finished:
            await UpdateTransactionAction(transaction=tx).run()

        if order.is_subscription:
            self.logger.context_push(customer_subscription_id=order.customer_subscription_id)
            self.logger.debug('Sync customer_subscription')
            await SyncCustomerSubscriptionAction(uid=self.uid, order_id=self.order_id).run()
        elif tx.finished:
            self.logger.debug('Last transaction is finished. Nothing to sync.')
