from typing import Optional

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.exceptions import OrderNotFoundError
from mail.payments.payments.storage.exceptions import OrderNotFound


class UpdateServiceDataServiceMerchantAction(AuthServiceMerchantMixin, BaseDBAction):
    """
    Updates order service_data. Used by services.
    """

    transact = True

    def __init__(self,
                 service_merchant_id: int,
                 order_id: int,
                 service_data: Optional[dict] = None,
                 service_tvm_id: Optional[int] = None,
                 service: Optional[Service] = None,
                 uid: Optional[int] = None,
                 ):
        super().__init__()
        self.service_merchant_id = service_merchant_id
        self.order_id = order_id
        self.service_data = service_data
        self.service_tvm_id = service_tvm_id
        self.service = service
        self.uid = uid

    async def handle(self) -> Order:
        assert self.uid is not None
        try:
            order = await self.storage.order.get(
                uid=self.uid,
                order_id=self.order_id,
                service_merchant_id=self.service_merchant_id,
                for_update=True,
            )
        except OrderNotFound:
            raise OrderNotFoundError
        order.data.service_data = self.service_data

        order = await self.storage.order.save(order)
        order = await self.storage.order.get(order.uid, order.order_id)

        assert order.order_id is not None
        await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()
        return order
