from datetime import datetime
from typing import AsyncGenerator, AsyncIterable, ClassVar, Dict, List, Optional, Tuple

from sendr_utils import alist

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.enums import MerchantRole, OrderKind, PayStatus, RefundStatus
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.report import Report
from mail.payments.payments.core.entities.task import Task
from mail.payments.payments.core.exceptions import (
    CoreFailError, CoreInteractionFatalError, ReportNotFoundError, ReportNotUploadedError
)
from mail.payments.payments.storage.exceptions import ReportNotFound
from mail.payments.payments.storage.mappers.order.order import SKIP_CONDITION, FindOrderParams
from mail.payments.payments.utils.helpers import create_csv_writer


class CreateReportAction(BaseDBAction):
    transact = True
    action_name = 'create_report_action'
    async_params = ('uid', 'lower_dt', 'upper_dt', 'report_id')
    retry_exceptions = (CoreInteractionFatalError,)

    KINDS_MAPPING: ClassVar[Dict[OrderKind, str]] = {
        OrderKind.PAY: "Заказ",
        OrderKind.REFUND: "Возврат",
        OrderKind.MULTI: "Мультизаказ"
    }

    REFUND_MAPPING: ClassVar[Dict[RefundStatus, str]] = {
        RefundStatus.COMPLETED: "Проведен",
        RefundStatus.CREATED: "Создан",
        RefundStatus.REQUESTED: "В процессе",
        RefundStatus.FAILED: "Отклонен",
    }

    PAY_MAPPING: ClassVar[Dict[PayStatus, str]] = {
        PayStatus.CANCELLED: "Отменен",
        PayStatus.HELD: "Ожидает подтверждения",
        PayStatus.IN_CANCEL: "В процессе отмены",
        PayStatus.IN_MODERATION: "В процессе модерации",
        PayStatus.IN_PROGRESS: "В процессе",
        PayStatus.MODERATION_NEGATIVE: "Не прошел модерацию",
        PayStatus.NEW: "Доступен к оплате",
        PayStatus.PAID: "Оплачен",
        PayStatus.REJECTED: "Доступен к оплате",
        PayStatus.ABANDONED: "Заброшен",
        PayStatus.VERIFIED: "Верифицирован"  # unused and deprecated
    }

    HEADER: List[str] = [
        "Номер строки", "Ключ товара", "Название товара", "Время выставления счета продавцом",
        "Время завершения оплаты", "Уникальный ключ заказа", "Почта покупателя", "Цена товара",
        "Количество", "Заказ или возврат", "Статус всего заказа/возврата", "Название заказа", "Способ оплаты"
    ]

    def __init__(self, uid: int, report_id: str):
        super().__init__()
        self.uid: int = uid
        self.report_id: str = report_id

    async def create_csv(self, lower_dt: datetime,
                         upper_dt: datetime,
                         pay_method: Optional[str] = None) -> AsyncGenerator[bytes, None]:
        writer, output = create_csv_writer()
        writer.writerow(self.HEADER)
        yield output.getvalue().encode('utf-8')

        row_id = 1
        orders: Dict[int, Order] = {
            order.order_id: order
            async for order in self.storage.order.find(FindOrderParams(
                uid=self.uid,
                created_from=lower_dt,
                created_to=upper_dt,
                pay_method=pay_method,
                exclude_stats=False,
                parent_order_id=SKIP_CONDITION
            )) if order.order_id is not None
        }

        uid_and_order_id_list = [(self.uid, order_id) for order_id in orders if order_id is not None]

        async for item in self.storage.item.get_for_orders(uid_and_order_id_list, iterator=True):
            assert item.order_id
            order = orders[item.order_id]
            status: str

            if order.kind == OrderKind.REFUND:
                assert order.refund_status is not None
                status = self.REFUND_MAPPING[order.refund_status]
            elif order.kind == OrderKind.PAY:
                assert order.pay_status is not None
                status = self.PAY_MAPPING[order.pay_status]
            else:
                continue  # Skipping multi-orders

            assert item.product is not None
            assert item.price is not None

            output.truncate(0)
            output.seek(0)
            writer.writerow([
                row_id,
                item.product.product_id,
                item.product.name,
                order.created or '-',
                order.closed or '-',
                order.order_id,
                order.user_email or '-',
                "%.2f" % item.price,
                "%.2f" % item.amount,
                self.KINDS_MAPPING[order.kind],
                status or '-',
                order.caption or '-',
                order.pay_method or '-'
            ])
            row_id += 1
            yield output.getvalue().encode('utf-8')

    async def handle(self) -> Report:
        try:
            report = await self.storage.report.get(report_id=self.report_id, for_update=True)
        except ReportNotFound:
            raise CoreFailError(f'Report for uid {self.uid} cannot be generated')

        assert report.data is not None
        lower_dt: datetime = report.data['lower_dt']
        upper_dt: datetime = report.data['upper_dt']
        pay_method: Optional[str] = report.data.get('pay_method')

        data = b"".join(await alist(self.create_csv(lower_dt=lower_dt, upper_dt=upper_dt, pay_method=pay_method)))
        report.mds_path = await self.clients.payments_mds.upload(f'uid_{self.uid}.report', data)
        return await self.storage.report.save(report)


class CreateTaskReportAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.VIEWER,)
    transact = True

    def __init__(self, uid: int, lower_dt: datetime, upper_dt: datetime, pay_method: Optional[str] = None):
        super().__init__()
        self.uid = uid
        self.lower_dt = lower_dt
        self.upper_dt = upper_dt
        self.pay_method = pay_method

    async def handle(self) -> Tuple[Task, Report]:
        report = await self.storage.report.create(
            Report(
                mds_path=None,
                uid=self.uid,
                data={
                    'lower_dt': self.lower_dt,
                    'upper_dt': self.upper_dt,
                    'pay_method': self.pay_method
                },
            )
        )
        assert report.report_id is not None
        task = await CreateReportAction(
            uid=self.uid,
            report_id=report.report_id,
        ).run_async()
        return task, report


class GetReportListAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.VIEWER,)

    def __init__(self, uid: int):
        super().__init__()
        self.uid: int = uid

    async def handle(self) -> List[Report]:
        return await alist(self.storage.report.find(uid=self.uid))


class DownloadReportAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.VIEWER,)

    def __init__(self, uid: int, report_id: str):
        super().__init__()
        self.uid: int = uid
        self.report_id: str = report_id

    async def handle(self) -> Tuple[Optional[str], AsyncIterable[bytes]]:
        try:
            report = await self.storage.report.get(report_id=self.report_id, uid=self.uid)
        except ReportNotFound:
            raise ReportNotFoundError(uid=self.uid, report_id=self.report_id)

        if not report.mds_path:
            raise ReportNotUploadedError
        content_type, data = await self.clients.payments_mds.download(report.mds_path, close=True)

        return content_type, data
