from typing import List, Optional

from sendr_utils import alist

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.service import Service, ServiceClient


class CreateServiceAction(BaseDBAction):
    transact = True

    def __init__(self,
                 name: str,
                 slug: str,
                 tvm_id: int,
                 api_callback_url: str,
                 order_moderation_enabled: bool = True,
                 antifraud: bool = False,
                 hidden: bool = False,
                 ):
        super().__init__()
        self.name: str = name
        self.slug: str = slug
        self.tvm_id: int = tvm_id
        self.api_callback_url: str = api_callback_url
        self.order_moderation_enabled: bool = order_moderation_enabled
        self.antifraud: bool = antifraud
        self.hidden: bool = hidden

    async def handle(self) -> Service:
        service = await self.storage.service.create(Service(
            name=self.name,
            slug=self.slug,
            order_moderation_enabled=self.order_moderation_enabled,
            antifraud=self.antifraud,
            hidden=self.hidden
        ))
        assert service.service_id
        await self.storage.service_client.create(ServiceClient(
            service_id=service.service_id,
            tvm_id=self.tvm_id,
            api_callback_url=self.api_callback_url,
        ))
        return service


class GetServiceListAction(BaseDBAction):
    def __init__(self,
                 hidden: Optional[bool] = None):
        super().__init__()
        self.hidden = hidden

    async def handle(self) -> List[Service]:
        services = await alist(self.storage.service.find(hidden=self.hidden))
        return services


class GetServiceListByServiceMerchantsAction(BaseDBAction):
    def __init__(self, uid: int):
        super().__init__()
        self.uid = uid

    async def handle(self) -> List[Service]:
        services = await alist(self.storage.service.find_by_service_merchants(self.uid))
        return services
