from typing import Optional, Tuple, cast

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.init_products import InitProductsAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.mixins.notify import NotifyMixin
from mail.payments.payments.core.actions.service_merchant.base import BaseServiceMerchantAction
from mail.payments.payments.core.context import MerchantUser
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import MerchantRole, OperationKind
from mail.payments.payments.core.entities.log import ServiceMerchantCreatedLog
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.service import ServiceMerchant
from mail.payments.payments.core.exceptions import (
    CoreActionDenyError, CoreFailError, ServiceMerchantAlreadyExists, ServiceNotFoundError
)
from mail.payments.payments.utils.helpers import temp_setattr


class CheckEnableServiceMerchantRolesAction(BaseDBAction):
    """Simple action allowing to check roles if autoenable
    """
    required_merchant_roles = (MerchantRole.ADMIN,)

    async def handle(self) -> None:
        pass


class CreateServiceMerchantAction(NotifyMixin, BaseServiceMerchantAction):
    """Used by service"""

    transact = True

    def __init__(self,
                 service_tvm_id: int,
                 entity_id: str,
                 description: str,
                 autoenable: bool = False,
                 uid: Optional[int] = None,
                 token: Optional[str] = None,
                 ):
        super().__init__()
        self.service_tvm_id: int = service_tvm_id
        self.entity_id: str = entity_id
        self.description: str = description
        self.autoenable: bool = autoenable
        self.uid: Optional[int] = uid
        self.token: Optional[str] = token

    async def handle(self) -> Tuple[ServiceMerchant, Merchant]:
        if self.uid is None and self.token is None:
            raise CoreFailError("Either UID or Token required")

        merchant: Merchant
        if self.token is not None:
            merchant = await GetMerchantAction(token=self.token).run()
        else:
            merchant = await GetMerchantAction(uid=self.uid).run()

        if not merchant.options.allow_create_service_merchants:
            raise CoreActionDenyError

        if self.autoenable and self.token:
            assert self.uid is not None
            merchant_user = MerchantUser(user_uid=self.uid, merchant_id=cast(str, merchant.merchant_id))
            with temp_setattr(self.context, 'merchant_user', merchant_user):
                await CheckEnableServiceMerchantRolesAction().run()

        if self.service_tvm_id:
            service = await self.get_service(service_tvm_id=self.service_tvm_id)
        else:
            raise ServiceNotFoundError(service_tvm_id=None)

        assert service.service_id
        service_merchant_entity = ServiceMerchant(
            uid=merchant.uid,
            service_id=service.service_id,
            entity_id=self.entity_id,
            description=self.description,
            enabled=self.autoenable,
        )
        service_merchant, created = await self.storage.service_merchant.get_or_create(
            service_merchant_entity, lookup_fields=('uid', 'service_id', 'entity_id')
        )
        if not created:
            if not service_merchant.enabled:
                async with self.storage_setter(transact=True):
                    await self.notify_merchant_on_sm_created_enabled(service=service,
                                                                     service_merchant=service_merchant,
                                                                     merchant=merchant)
            raise ServiceMerchantAlreadyExists(service_merchant_id=service_merchant.service_merchant_id)

        if (service_fee := service.options.service_fee) is not None:
            await InitProductsAction(merchant=merchant, service_fee=service_fee).run()

        service_merchant.service = service
        await self.notify_merchant_on_sm_created_enabled(service=service,
                                                         service_merchant=service_merchant,
                                                         merchant=merchant)
        # Logging
        assert service_merchant.revision and service_merchant.service_merchant_id
        await self.storage.change_log.create(ChangeLog(
            uid=merchant.uid,
            revision=service_merchant.revision,
            operation=OperationKind.ADD_SERVICE_MERCHANT,
            arguments={
                'service_id': service_merchant.service_id,
                'service_merchant_id': service_merchant.service_merchant_id
            }
        ))

        await self.pushers.log.push(ServiceMerchantCreatedLog(
            merchant_uid=merchant.uid,
            service_id=service.service_id,
            service_name=service.name,
            service_merchant_id=service_merchant.service_merchant_id,
            enabled=service_merchant.enabled,
        ))
        self.logger.info('Service_merchant created.')

        return service_merchant, merchant
