from typing import Optional

from sendr_utils import alist

from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.service_merchant.base import BaseServiceMerchantAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import MerchantRole, OperationKind
from mail.payments.payments.core.entities.log import ServiceMerchantUpdatedLog
from mail.payments.payments.core.entities.service import ServiceMerchant


class UpdateServiceMerchantAction(APICallbackTaskMixin, BaseServiceMerchantAction):
    """Used by merchant"""
    transact = True
    required_merchant_roles = (MerchantRole.ADMIN,)

    def __init__(self,
                 uid: int,
                 service_merchant_id: int,
                 enabled: Optional[bool] = None,
                 description: Optional[str] = None,
                 ):
        super().__init__()
        self.uid: int = uid
        self.service_merchant_id: int = service_merchant_id
        self.enabled: Optional[bool] = enabled
        self.description: Optional[str] = description

    async def handle(self) -> ServiceMerchant:
        service_merchant = await self.get_service_merchant(
            service_merchant_id=self.service_merchant_id,
            uid=self.uid,
        )
        if self.enabled is None and self.description is None:
            return service_merchant
        if self.enabled is not None:
            service_merchant.enabled = self.enabled
        if self.description is not None:
            service_merchant.description = self.description

        service_merchant = await self.storage.service_merchant.save(service_merchant)
        service_merchant.service = await self.storage.service.get(service_merchant.service_id)

        # Logging
        assert service_merchant.revision and service_merchant.service_merchant_id
        await self.storage.change_log.create(ChangeLog(
            uid=self.uid,
            revision=service_merchant.revision,
            operation=OperationKind.UPDATE_SERVICE_MERCHANT,
            arguments={'service_id': service_merchant.service_id,
                       'service_merchant_id': service_merchant.service_merchant_id}
        ))
        self.logger.info('Service_merchant updated')
        await self.pushers.log.push(ServiceMerchantUpdatedLog(
            merchant_uid=service_merchant.uid,
            service_id=service_merchant.service_id,
            service_name=service_merchant.service.name,
            service_merchant_id=service_merchant.service_merchant_id,
            enabled=service_merchant.enabled,
        ))
        service_clients = await alist(self.storage.service_client.find(service_id=service_merchant.service_id))
        for service_client in service_clients:
            service = await self.storage.service.get_by_related(
                service_merchant_id=service_merchant.service_merchant_id,
                service_client_id=service_client.service_client_id
            )
            await self.create_service_merchant_updated_task(service)

        return service_merchant
