from typing import Any, Optional

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.enums import MerchantRole, ShopType
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.shop import Shop
from mail.payments.payments.core.exceptions import ShopDefaultMustBePresent, ShopNotFoundError
from mail.payments.payments.storage.exceptions import ShopNotFound


class CoreCreateOrUpdateShopAction(BaseMerchantAction):
    skip_data = True
    skip_moderation = True
    skip_parent = True
    skip_oauth = True

    for_update = True
    transact = True

    def __init__(self,
                 name: str,
                 shop_type: Optional[ShopType] = None,
                 is_default: bool = False,
                 shop_id: Optional[int] = None,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None):
        super().__init__(uid=uid, merchant=merchant)
        self.name = name
        self.shop_type = shop_type
        self.shop_id = shop_id
        self.is_default = is_default

    async def handle(self) -> Shop:
        assert self.merchant

        if self.shop_id is not None:
            try:
                shop = await self.storage.shop.get(uid=self.merchant.uid, shop_id=self.shop_id, for_update=True)
            except ShopNotFound:
                raise ShopNotFoundError

            if shop.is_default and not self.is_default:
                raise ShopDefaultMustBePresent(params={'shop_type': shop.shop_type.value})

            shop.name = self.name
            shop.is_default = self.is_default
        else:
            assert self.shop_type
            shop = Shop(uid=self.merchant.uid, name=self.name, is_default=self.is_default, shop_type=self.shop_type)

        try:
            default_shop: Optional[Shop] = await self.storage.shop.get_default_for_merchant(
                shop.uid,
                shop.shop_type,
                for_update=True
            )
        except ShopNotFound:
            default_shop = None
            shop.is_default = True

        if shop.is_default and default_shop is not None and shop.shop_id != default_shop.shop_id:
            default_shop.is_default = False
            await self.storage.shop.save(default_shop)

        if shop.shop_id:
            shop = await self.storage.shop.save(shop)
        else:
            shop = await self.storage.shop.create(shop)

        return shop


class CreateOrUpdateShopAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)
    transact = True

    def __init__(self, **kwargs: Any):
        super().__init__()
        self.kwargs = kwargs

    async def handle(self) -> Shop:
        return await CoreCreateOrUpdateShopAction(**self.kwargs).run()
