from typing import Optional

from psycopg2.errors import UniqueViolation  # NOQA

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.enums import ShopType
from mail.payments.payments.core.entities.shop import Shop
from mail.payments.payments.core.exceptions import ShopNotFoundError
from mail.payments.payments.storage.exceptions import ShopNotFound


class GetOrEnsureDefaultShopAction(BaseDBAction):
    """
    Get Shop or ensure default shop exists (get or create default shop) of given type.
    Given shop_id just get shop by uid, shop_id pair, otherwise
    get default merchant shop, and if it does not exist create one.
    """

    transact = False

    def __init__(self, uid: int, shop_id: Optional[int] = None, default_shop_type: ShopType = ShopType.PROD):
        super().__init__()
        self.uid = uid
        self.shop_id = shop_id
        self.default_shop_type = default_shop_type

    async def handle(self) -> Shop:
        if self.shop_id is not None:
            try:
                return await self.storage.shop.get(uid=self.uid, shop_id=self.shop_id)
            except ShopNotFound:
                raise ShopNotFoundError(
                    message='Shop not found',
                    params={'shop_id': self.shop_id},
                )

        default_shop_entity = Shop(uid=self.uid, name='Основной', is_default=True, shop_type=self.default_shop_type)
        if default_shop_entity.shop_type == ShopType.TEST:
            default_shop_entity.name += ' тестовый'

        shop, _ = await self.storage.shop.get_or_create(
            default_shop_entity,
            lookup_fields=('uid', 'shop_type', 'is_default'),
        )
        return shop
