from typing import ClassVar, Optional

from sendr_utils import alist

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.enums import ModerationType
from mail.payments.payments.core.entities.moderation import Moderation, ModerationData
from mail.payments.payments.core.entities.subscription import Subscription
from mail.payments.payments.core.exceptions import CoreActionDenyError, SubscriptionNotFoundError
from mail.payments.payments.storage.exceptions import MerchantNotFound, SubscriptionNotFound


class BaseSubscriptionAction(BaseDBAction):
    subscription_for_update: ClassVar[bool] = False
    subscription_allow_none: ClassVar[bool] = False
    subscription_manual_load: ClassVar[bool] = False
    subscription_skip_moderation: ClassVar[bool] = False
    subscription_check_moderation_approved: ClassVar[bool] = False

    uid: Optional[int] = None
    subscription: Optional[Subscription] = None
    _subscription_id: Optional[int] = None

    def __init__(self,
                 uid: Optional[int] = None,
                 subscription_id: Optional[int] = None,
                 ):
        super().__init__()
        self.uid = uid
        self._subscription_id = subscription_id

    @property
    def subscription_id(self):
        assert self.subscription
        return self.subscription.subscription_id

    async def _has_ongoing_subscription_moderations(self, subscription: Subscription) -> bool:
        if subscription.moderation is not None and subscription.moderation.has_ongoing is not None:
            return subscription.moderation.has_ongoing
        ongoing = await alist(
            self.storage.moderation.find(uid=subscription.uid,
                                         entity_id=subscription.subscription_id,
                                         moderation_type=ModerationType.SUBSCRIPTION,
                                         has_approved=False,
                                         ignore=False)
        )
        return bool(ongoing)

    async def _get_effective_subscription_moderation(self, subscription: Subscription) -> Optional[Moderation]:
        return await self.storage.moderation.get_effective(uid=subscription.uid,
                                                           entity_id=subscription.subscription_id,
                                                           moderation_type=ModerationType.SUBSCRIPTION)

    async def get_moderation_subscription_data(self, subscription: Subscription) -> ModerationData:
        effective = await self._get_effective_subscription_moderation(subscription)
        has_ongoing = await self._has_ongoing_subscription_moderations(subscription)

        return ModerationData(
            approved=effective.approved if effective is not None else settings.SUBSCRIPTION_MODERATION_DISABLED,
            reasons=effective.reasons if effective is not None else [],
            has_moderation=has_ongoing or effective is not None,
            has_ongoing=has_ongoing,
        )

    async def _load_subscription(self, uid: Optional[int] = None, subscription_id: Optional[int] = None) -> None:
        try:
            if uid is not None and subscription_id is not None:
                self.subscription = await self.storage.subscription.get(uid,
                                                                        subscription_id,
                                                                        for_update=self.subscription_for_update)
            else:
                raise RuntimeError('No subscription or uid and subscription_id is provided')
        except SubscriptionNotFound:
            pass

        if self.subscription is None:
            if self.subscription_allow_none:
                return
            raise SubscriptionNotFoundError(uid=uid, subscription_id=self._subscription_id)

        if not self.subscription_skip_moderation:
            await self._load_subscription_moderation()

    async def _load_subscription_moderation(self) -> None:
        assert self.subscription
        self.subscription.moderation = await self.get_moderation_subscription_data(self.subscription)

    async def approved_subscription_moderation(self, subscription: Subscription) -> bool:
        if subscription.moderation is not None:
            return bool(subscription.moderation.approved)
        moderation = await self.get_moderation_subscription_data(subscription)
        return moderation is not None and bool(moderation.approved)

    async def load_subscription(self, uid: int, subscription_id: Optional[int] = None) -> None:
        await self._load_subscription(uid=uid, subscription_id=subscription_id)

        self.logger.context_push(
            uid=self.subscription.uid if self.subscription is not None else None,
            subscription_id=self.subscription.subscription_id if self.subscription is not None else None
        )

        if self.subscription_check_moderation_approved:
            try:
                merchant = await self.storage.merchant.get(uid=uid)
            except MerchantNotFound:
                raise CoreActionDenyError

            if not merchant.trustworthy and (
                self.subscription is None or not await self.approved_subscription_moderation(self.subscription)
            ):
                raise CoreActionDenyError

    async def pre_handle(self) -> None:
        if not self.subscription_manual_load:
            assert self.uid is not None
            await self.load_subscription(uid=self.uid, subscription_id=self._subscription_id)
        await super().pre_handle()
