import asyncio
from itertools import count
from typing import List, Optional

from sendr_interactions import exceptions as interaction_errors

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.mixins.moderation import MerchantModerationMixin
from mail.payments.payments.core.actions.moderation import ScheduleSubscriptionModerationAction
from mail.payments.payments.core.actions.subscription.base import BaseSubscriptionAction
from mail.payments.payments.core.entities.enums import (
    NDS, FunctionalityType, MerchantOAuthMode, MerchantRole, PeriodUnit, ShopType
)
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.subscription import Subscription, SubscriptionData, SubscriptionPrice
from mail.payments.payments.core.exceptions import CoreDataError, CoreFieldError
from mail.payments.payments.interactions import GeobaseClient


class CoreCreateSubscriptionAction(MerchantModerationMixin, BaseSubscriptionAction):
    transact = True
    subscription_manual_load = True

    def __init__(self,
                 uid: int,
                 title: str,
                 fiscal_title: str,
                 nds: NDS,
                 period_amount: int,
                 period_units: PeriodUnit,
                 prices: List[dict],
                 trial_period_amount: Optional[int] = None,
                 trial_period_units: Optional[PeriodUnit] = None,
                 merchant_oauth_mode: Optional[MerchantOAuthMode] = None,
                 merchant: Optional[Merchant] = None,
                 service_merchant_id: Optional[int] = None,
                 service_client_id: Optional[int] = None,
                 fast_moderation: bool = False,
                 ):
        super().__init__(uid=uid)
        self.title = title
        self.fiscal_title = fiscal_title
        self.nds = nds
        self.period_amount = period_amount
        self.period_units = period_units
        self.prices = [
            SubscriptionPrice(
                price=price['price'],
                region_id=price['region_id'],
                currency=price['currency']
            )
            for price in prices
        ]
        self.trial_period_amount = trial_period_amount
        self.trial_period_units = trial_period_units
        self.merchant_oauth_mode: MerchantOAuthMode = merchant_oauth_mode or MerchantOAuthMode.PROD
        self.merchant = merchant
        self.service_merchant_id = service_merchant_id
        self.service_client_id = service_client_id
        self.fast_moderation = fast_moderation

    async def pre_handle(self) -> None:
        await super().pre_handle()

        self.merchant = await GetMerchantAction(
            uid=self.uid,
            merchant=self.merchant,
            skip_data=True,
        ).run()
        await self.require_moderation(self.merchant, functionality_type=FunctionalityType.PAYMENTS)

        region_ids = [price.region_id for price in self.prices]
        geo_responses = await asyncio.gather(*[self.clients.geobase.get_region(region_id) for region_id in region_ids],
                                             return_exceptions=True)

        for num, region_id, geo_response in zip(count(), region_ids, geo_responses):
            field = f"prices.{num}.region_id"
            if isinstance(geo_response, interaction_errors.BaseInteractionError):
                raise CoreFieldError(fields={field: f"Unknown region_id: {region_id}"})
            elif geo_response['type'] != GeobaseClient.REGION_TYPE_COUNTRY:
                raise CoreFieldError(fields={field: "region_id must be id of country"})

    async def handle(self) -> Subscription:
        assert (
            self.uid is not None
            and self.merchant is not None
        )
        if self.merchant.acquirer is None:
            raise CoreDataError(message='Can not create subscription for merchant without acquirer')

        subscription_data = SubscriptionData(
            fast_moderation=self.fast_moderation,
        )

        subscription_entity = Subscription(
            uid=self.uid,
            service_merchant_id=self.service_merchant_id,
            service_client_id=self.service_client_id,
            title=self.title,
            fiscal_title=self.fiscal_title,
            nds=self.nds,
            period_amount=self.period_amount,
            period_units=self.period_units,
            prices=self.prices,
            trial_period_amount=self.trial_period_amount,
            trial_period_units=self.trial_period_units,
            merchant_oauth_mode=self.merchant_oauth_mode,
            acquirer=self.merchant.acquirer,
            data=subscription_data,
        )

        if subscription_entity.approx_period_seconds < settings.CUSTOMER_SUBSCRIPTION_MIN_PERIOD:
            raise CoreFieldError(
                fields={
                    'period_amount': f'period should be more then {settings.CUSTOMER_SUBSCRIPTION_MIN_PERIOD} seconds'
                }
            )

        self.subscription = await self.storage.subscription.create(subscription_entity)

        shop_type = ShopType.from_oauth_mode(self.subscription.merchant_oauth_mode)
        trust = self.clients.get_trust_client(self.subscription.uid, shop_type)
        assert self.subscription.acquirer
        await trust.product_subscription_create(
            uid=self.subscription.uid,
            acquirer=self.subscription.acquirer,
            merchant=self.merchant,
            subscription=self.subscription,
        )

        self.logger.context_push(subscription_id=self.subscription.subscription_id)
        self.logger.info('Subscription created')

        if not settings.SUBSCRIPTION_MODERATION_DISABLED:
            await ScheduleSubscriptionModerationAction(subscription=self.subscription).run()
        await self._load_subscription_moderation()

        return self.subscription


class CreateSubscriptionAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)

    def __init__(self,
                 uid: int,
                 title: str,
                 fiscal_title: str,
                 nds: NDS,
                 period_amount: int,
                 period_units: PeriodUnit,
                 prices: List[dict],
                 trial_period_amount: Optional[int] = None,
                 trial_period_units: Optional[PeriodUnit] = None,
                 merchant_oauth_mode: Optional[MerchantOAuthMode] = None,
                 fast_moderation: bool = False,
                 ):
        super().__init__()
        self.uid = uid
        self.title = title
        self.fiscal_title = fiscal_title
        self.nds = nds
        self.period_amount = period_amount
        self.period_units = period_units
        self.prices = prices
        self.trial_period_amount = trial_period_amount
        self.trial_period_units = trial_period_units
        self.merchant_oauth_mode: MerchantOAuthMode = merchant_oauth_mode or MerchantOAuthMode.PROD
        self.fast_moderation = fast_moderation

    async def handle(self) -> Subscription:
        return await CoreCreateSubscriptionAction(uid=self.uid,
                                                  title=self.title,
                                                  fiscal_title=self.fiscal_title,
                                                  nds=self.nds,
                                                  period_amount=self.period_amount,
                                                  period_units=self.period_units,
                                                  prices=self.prices,
                                                  trial_period_amount=self.trial_period_amount,
                                                  trial_period_units=self.trial_period_units,
                                                  merchant_oauth_mode=self.merchant_oauth_mode,
                                                  fast_moderation=self.fast_moderation,
                                                  ).run()


class CreateSubscriptionServiceMerchantAction(AuthServiceMerchantMixin, BaseDBAction):
    def __init__(self,
                 service_merchant_id: int,
                 service_tvm_id: int,
                 title: str,
                 fiscal_title: str,
                 nds: NDS,
                 period_amount: int,
                 period_units: PeriodUnit,
                 prices: List[dict],
                 trial_period_amount: Optional[int] = None,
                 trial_period_units: Optional[PeriodUnit] = None,
                 merchant_oauth_mode: Optional[MerchantOAuthMode] = None,
                 fast_moderation: bool = False,
                 ):
        super().__init__()
        self.service_merchant_id = service_merchant_id
        self.service_tvm_id = service_tvm_id
        self.title = title
        self.fiscal_title = fiscal_title
        self.nds = nds
        self.period_amount = period_amount
        self.period_units = period_units
        self.prices = prices
        self.trial_period_amount = trial_period_amount
        self.trial_period_units = trial_period_units
        self.merchant_oauth_mode: MerchantOAuthMode = merchant_oauth_mode or MerchantOAuthMode.PROD
        self.fast_moderation = fast_moderation

    async def handle(self) -> Subscription:
        assert self.uid
        return await CoreCreateSubscriptionAction(uid=self.uid,
                                                  title=self.title,
                                                  fiscal_title=self.fiscal_title,
                                                  nds=self.nds,
                                                  period_amount=self.period_amount,
                                                  period_units=self.period_units,
                                                  prices=self.prices,
                                                  trial_period_amount=self.trial_period_amount,
                                                  trial_period_units=self.trial_period_units,
                                                  merchant_oauth_mode=self.merchant_oauth_mode,
                                                  service_merchant_id=self.service_merchant_id,
                                                  service_client_id=self.service_client_id,
                                                  fast_moderation=self.fast_moderation,
                                                  ).run()
