from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.order.get import GetOrderByCustomerSubscriptionIdAction
from mail.payments.payments.core.actions.tlog.schemas import subscription_tlog_schema
from mail.payments.payments.core.actions.tlog.write import WriteToTLogAction
from mail.payments.payments.core.entities.enums import PayStatus, TransactionStatus
from mail.payments.payments.core.exceptions import OrderNotFoundError
from mail.payments.payments.interactions.trust.base import BaseTrustClient
from mail.payments.payments.storage.exceptions import CustomerSubscriptionTransactionNotFound


class ExportCustomerSubscriptionToTLogAction(BaseDBAction):
    action_name = 'export_customer_subscription_transaction_to_tlog'
    max_retries = 100

    def __init__(self, uid: int, customer_subscription_id: int, purchase_token: str):
        super().__init__()
        self.uid = uid
        self.customer_subscription_id = customer_subscription_id
        self.purchase_token = purchase_token

    async def handle(self) -> None:
        self.logger.context_push(uid=self.uid,
                                 customer_subscription_id=self.customer_subscription_id,
                                 purchase_token=self.purchase_token)

        try:
            tx = await self.storage.customer_subscription_transaction.get(self.uid,
                                                                          self.customer_subscription_id,
                                                                          self.purchase_token)
        except CustomerSubscriptionTransactionNotFound:
            self.logger.info('CustomerSubscriptionTransaction not found')
            return

        try:
            order = await GetOrderByCustomerSubscriptionIdAction(uid=tx.uid,
                                                                 customer_subscription_id=self.customer_subscription_id,
                                                                 skip_add_crypto=True).run()
        except OrderNotFoundError:
            self.logger.info('Order not found')
            return

        if tx.payment_status != TransactionStatus.CLEARED:
            self.logger.warning('CustomerSubscriptionTransaction is not cleared')
            return
        elif order.pay_status != PayStatus.PAID:
            self.logger.warning(f'Pay status of order is not {PayStatus.PAID.value}')
            return
        elif order.is_test:
            self.logger.info('CustomerSubscription is test, skip')
            return

        merchant = await GetMerchantAction(uid=order.uid).run()
        order.acquirer = order.get_acquirer(merchant.acquirer)
        for item in order.items:
            item.trust_order_id = BaseTrustClient.make_order_id(
                order.uid, order.order_id, item.product_id, order.customer_uid, order.data.version
            )

        await WriteToTLogAction(data={
            'order': order,
            'merchant': merchant,
            'customer_subscription_transaction': tx
        }, schema=subscription_tlog_schema).run()
