from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.order.get import CoreGetOrderAction
from mail.payments.payments.core.actions.tlog.schemas import order_tlog_schema
from mail.payments.payments.core.actions.tlog.write import WriteToTLogAction
from mail.payments.payments.core.entities.enums import PAY_METHOD_YANDEX, OrderKind, PayStatus, TransactionStatus
from mail.payments.payments.interactions.trust.base import BaseTrustClient
from mail.payments.payments.storage.exceptions import OrderNotFound, TransactionNotFound


class ExportOrderToTLogAction(BaseDBAction):
    action_name = 'export_order_transaction_to_tlog'
    max_retries = 100

    def __init__(self, uid: int, tx_id: int):
        super().__init__()
        self.uid = uid
        self.tx_id = tx_id

    async def handle(self) -> None:
        self.logger.context_push(uid=self.uid, tx_id=self.tx_id)

        try:
            tx = await self.storage.transaction.get(uid=self.uid, tx_id=self.tx_id)
        except TransactionNotFound:
            self.logger.info('Transaction not found')
            return

        try:
            order = await CoreGetOrderAction(uid=tx.uid,
                                             order_id=tx.order_id,
                                             kind=OrderKind.PAY,
                                             skip_add_crypto=True).run()
        except OrderNotFound:
            self.logger.info('Order not found')
            return

        if tx.status != TransactionStatus.CLEARED:
            self.logger.warning('Transaction is not cleared')
            return
        elif order.pay_method != PAY_METHOD_YANDEX:
            self.logger.warning(f'Pay method of order is not {PAY_METHOD_YANDEX}')
            return
        elif order.pay_status != PayStatus.PAID:
            self.logger.warning(f'Pay status of order is not {PayStatus.PAID.value}')
            return
        elif order.is_test:
            self.logger.info('Order is test, skip')
            return

        merchant = await GetMerchantAction(uid=order.uid).run()

        order.acquirer = order.get_acquirer(merchant.acquirer)
        for item in order.items:
            item.trust_order_id = BaseTrustClient.make_order_id(
                order.uid, order.order_id, item.product_id, order.customer_uid, order.data.version
            )

        await WriteToTLogAction(data={
            'order': order,
            'transaction': tx,
            'merchant': merchant
        }, schema=order_tlog_schema).run()
