from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.order.get import CoreGetOrderAction
from mail.payments.payments.core.actions.tlog.schemas import refund_tlog_schema
from mail.payments.payments.core.actions.tlog.write import WriteToTLogAction
from mail.payments.payments.core.entities.enums import OrderKind, RefundStatus
from mail.payments.payments.core.exceptions import OrderNotFoundError
from mail.payments.payments.interactions.trust.base import BaseTrustClient
from mail.payments.payments.storage.exceptions import TransactionNotFound


class ExportRefundToTLogAction(BaseDBAction):
    action_name = 'export_refund_to_tlog'
    max_retries = 100

    def __init__(self, uid: int, order_id: int):
        super().__init__()
        self.uid = uid
        self.order_id = order_id

    async def handle(self) -> None:
        try:
            order = await CoreGetOrderAction(
                uid=self.uid,
                order_id=self.order_id,
                kind=OrderKind.REFUND,
                skip_add_crypto=True
            ).run()
        except OrderNotFoundError:
            self.logger.info('Order not found')
            return

        if order.refund_status != RefundStatus.COMPLETED:
            self.logger.warning('Refund is not completed')
            return
        elif order.is_test:
            self.logger.info('Refund is test, skip')
            return

        try:
            original_order = await CoreGetOrderAction(
                uid=self.uid,
                order_id=order.original_order_id,
                kind=OrderKind.PAY,
                skip_add_crypto=True
            ).run()
        except OrderNotFoundError:
            self.logger.warning('Original order not found')
            return

        try:
            original_order_transaction = await self.storage.transaction.get_last_by_order(
                uid=original_order.uid,
                order_id=original_order.order_id,
            )
        except TransactionNotFound:
            self.logger.warning('Original transaction not found')
            return

        merchant = await GetMerchantAction(uid=order.uid).run()
        order.acquirer = order.get_acquirer(merchant.acquirer)
        for item in order.items:
            item.trust_order_id = BaseTrustClient.make_order_id(
                order.uid, order.order_id, item.product_id, order.customer_uid, order.data.version
            )

        original_order.acquirer = original_order.get_acquirer(merchant.acquirer)
        for item in original_order.items:
            item.trust_order_id = BaseTrustClient.make_order_id(
                original_order.uid, original_order.order_id, item.product_id,
                original_order.customer_uid, original_order.data.version
            )

        await WriteToTLogAction(data={
            'refund': order,
            'original_order': original_order,
            'original_order_transaction': original_order_transaction,
            'merchant': merchant,
        }, schema=refund_tlog_schema).run()
