from marshmallow import Schema, fields
from marshmallow_enum import EnumField

from mail.payments.payments.core.entities.enums import NDS, AcquirerType, OrderKind


# Base
class BaseSchema(Schema):
    class Meta:
        strict = True


# Entities
class ItemSchema(BaseSchema):
    name = fields.String()
    currency = fields.String()
    product_id = fields.Integer()
    nds = EnumField(NDS, by_value=True)
    amount = fields.Decimal(places=2)
    prices = fields.Decimal(places=2)
    total_price = fields.Decimal(places=2)
    markup = fields.Dict(default=None, missing=None)

    payment_method = fields.Constant("card")
    trust_order_id = fields.String()


class TransactionSchema(BaseSchema):
    tx_id = fields.Integer()
    revision = fields.Integer()
    created = fields.DateTime()
    updated = fields.DateTime()

    trust_purchase_token = fields.String()
    trust_resp_code = fields.String()
    trust_payment_id = fields.String()
    trust_terminal_id = fields.Integer()


class CustomerSubscriptionTransactionSchema(BaseSchema):
    created = fields.DateTime()
    updated = fields.DateTime()

    trust_purchase_token = fields.String(attribute='purchase_token')
    trust_terminal_id = fields.Integer(attribute='data.terminal.id')


class ServiceSchema(BaseSchema):
    service_fee = fields.Integer(attribute='options.service_fee')


class ServiceMerchantSchema(BaseSchema):
    service_merchant_id = fields.Integer()
    service_id = fields.Integer()
    service = fields.Nested(ServiceSchema)


class OrderSchema(BaseSchema):
    uid = fields.Integer()
    shop_id = fields.Integer()
    order_id = fields.Integer()
    parent_order_id = fields.Integer()
    original_order_id = fields.Integer()
    revision = fields.Integer()
    acquirer = EnumField(AcquirerType, by_value=True)

    commission = fields.Integer()
    kind = EnumField(OrderKind, by_value=True)

    autoclear = fields.Boolean()

    closed = fields.DateTime()
    created = fields.DateTime()
    updated = fields.DateTime()
    held_at = fields.DateTime()
    pay_status_updated_at = fields.DateTime()

    caption = fields.String()
    description = fields.String()

    customer_uid = fields.Integer()

    price = fields.Decimal(places=2)
    currency = fields.String()

    items = fields.Nested(ItemSchema, many=True)

    service_client_id = fields.Integer()
    service_merchant = fields.Nested(ServiceMerchantSchema)


class RefundSchema(OrderSchema):
    trust_refund_id = fields.String()


class MerchantSchema(BaseSchema):
    uid = fields.Integer()
    revision = fields.Integer()

    client_id = fields.String()
    person_id = fields.String()
    contract_id = fields.String()
    submerchant_id = fields.String()

    parent_uid = fields.Integer()


# TLog
class BaseTLogSchema(BaseSchema):
    merchant = fields.Nested(MerchantSchema)


class OrderTLogSchema(BaseTLogSchema):
    type_ = fields.Constant("payment", dump_to="type", dump_only=True)

    order = fields.Nested(OrderSchema)
    transaction = fields.Nested(TransactionSchema)


class RefundTLogSchema(OrderTLogSchema):
    type_ = fields.Constant("refund", dump_to="type", dump_only=True)

    refund = fields.Nested(RefundSchema)
    original_order = fields.Nested(OrderSchema)
    original_order_transaction = fields.Nested(TransactionSchema)


class SubscriptionTLogSchema(BaseTLogSchema):
    type_ = fields.Constant("subscription", dump_to="type", dump_only=True)

    customer_subscription_transaction = fields.Nested(CustomerSubscriptionTransactionSchema)
    order = fields.Nested(OrderSchema)


order_tlog_schema = OrderTLogSchema()
refund_tlog_schema = RefundTLogSchema()
subscription_tlog_schema = SubscriptionTLogSchema()
